# Document Management System - Dashboard Feature

## Feature Document v1.0
**Last Updated:** December 20, 2025  
**Platform:** CodeIgniter 3 + PostgreSQL  
**Author:** Development Team

---

## Table of Contents
1. [Feature Overview](#feature-overview)
2. [Architecture Diagram](#architecture-diagram)
3. [Database Schema](#database-schema)
4. [Component Details](#component-details)
5. [Business Logic](#business-logic)
6. [API Endpoints](#api-endpoints)
7. [Runbook](#runbook)
8. [Troubleshooting Guide](#troubleshooting-guide)

---

## Feature Overview

### Purpose
The Dashboard provides a centralized view for document compliance management, allowing users to track document upload status across multiple companies, authorities, and document types.

### Key Features
- **5-Tab Interface**: Favourite, Pending, Overdue, Upcoming, Uploaded
- **Real-time Status Tracking**: Visual indicators for document upload status
- **Quick Upload Navigation**: Direct upload from dashboard with pre-filled parameters
- **Favourite Management**: Mark frequently accessed documents for quick access
- **Period-based Filtering**: Documents organized by submission periods

### User Benefits
- At-a-glance compliance status
- Reduced missed deadlines
- Streamlined document upload workflow
- Prioritized action items

---

## Architecture Diagram

```
┌─────────────────────────────────────────────────────────────────────────────┐
│                              PRESENTATION LAYER                              │
├─────────────────────────────────────────────────────────────────────────────┤
│                                                                              │
│  ┌──────────────────────────────────────────────────────────────────────┐   │
│  │                         Dashboard View                                │   │
│  │                    (application/views/dashboard.php)                  │   │
│  ├──────────────────────────────────────────────────────────────────────┤   │
│  │  ┌──────────┐ ┌──────────┐ ┌──────────┐ ┌──────────┐ ┌──────────┐   │   │
│  │  │Favourite │ │ Pending  │ │ Overdue  │ │ Upcoming │ │ Uploaded │   │   │
│  │  │   Tab    │ │   Tab    │ │   Tab    │ │   Tab    │ │   Tab    │   │   │
│  │  └────┬─────┘ └────┬─────┘ └────┬─────┘ └────┬─────┘ └────┬─────┘   │   │
│  │       │            │            │            │            │          │   │
│  │       └────────────┴────────────┴────────────┴────────────┘          │   │
│  │                              │                                        │   │
│  │                    ┌─────────▼─────────┐                             │   │
│  │                    │   Upload Button   │ ──────────────────────┐     │   │
│  │                    │  (Per Row Action) │                       │     │   │
│  │                    └───────────────────┘                       │     │   │
│  └──────────────────────────────────────────────────────────────────────┘   │
│                                                                    │         │
│  ┌────────────────────────────────────────────────────────────────▼─────┐   │
│  │                       Upload Document View                           │   │
│  │                (application/views/company/upload_view.php)           │   │
│  │  ┌─────────────────────────────────────────────────────────────┐    │   │
│  │  │  Pre-filled: Company | Authority | Type | Month | Year      │    │   │
│  │  └─────────────────────────────────────────────────────────────┘    │   │
│  └──────────────────────────────────────────────────────────────────────┘   │
│                                                                              │
└──────────────────────────────────────────────────────────────────────────────┘
                                      │
                                      │ HTTP/AJAX
                                      ▼
┌─────────────────────────────────────────────────────────────────────────────┐
│                              CONTROLLER LAYER                                │
├─────────────────────────────────────────────────────────────────────────────┤
│                                                                              │
│  ┌────────────────────────────┐    ┌────────────────────────────┐           │
│  │   Dashboard Controller     │    │    Upload Controller       │           │
│  │  (controllers/dashboard.php)│    │  (controllers/upload.php)  │           │
│  ├────────────────────────────┤    ├────────────────────────────┤           │
│  │ • index()                  │    │ • index()                  │           │
│  │ • toggleFavourite()        │    │ • getTypes()               │           │
│  │ • getTabData()             │    │ • getDocuments()           │           │
│  │ • getCounts()              │    │ • upload()                 │           │
│  └─────────────┬──────────────┘    └─────────────┬──────────────┘           │
│                │                                  │                          │
└────────────────┼──────────────────────────────────┼──────────────────────────┘
                 │                                  │
                 │ Model Calls                      │
                 ▼                                  ▼
┌─────────────────────────────────────────────────────────────────────────────┐
│                                MODEL LAYER                                   │
├─────────────────────────────────────────────────────────────────────────────┤
│                                                                              │
│  ┌────────────────────────────────────────────────────────────────────┐     │
│  │                        Dashboard_model                              │     │
│  │                  (models/Dashboard_model.php)                       │     │
│  ├────────────────────────────────────────────────────────────────────┤     │
│  │                                                                     │     │
│  │  ┌─────────────────────┐  ┌─────────────────────┐                  │     │
│  │  │   getFavourites()   │  │ getPendingDocuments()│                  │     │
│  │  └─────────────────────┘  └─────────────────────┘                  │     │
│  │                                                                     │     │
│  │  ┌─────────────────────┐  ┌─────────────────────┐                  │     │
│  │  │getOverdueDocuments()│  │getUpcomingDocuments()│                  │     │
│  │  └─────────────────────┘  └─────────────────────┘                  │     │
│  │                                                                     │     │
│  │  ┌─────────────────────┐  ┌─────────────────────┐                  │     │
│  │  │getUploadedDocuments()│  │  toggleFavourite() │                  │     │
│  │  └─────────────────────┘  └─────────────────────┘                  │     │
│  │                                                                     │     │
│  │  ┌─────────────────────┐  ┌─────────────────────┐                  │     │
│  │  │ getDashboardCounts()│  │_getPreviousMonthsOverdue()│            │     │
│  │  └─────────────────────┘  └─────────────────────┘                  │     │
│  │                                                                     │     │
│  └────────────────────────────────────────────────────────────────────┘     │
│                                      │                                       │
└──────────────────────────────────────┼───────────────────────────────────────┘
                                       │
                                       │ SQL Queries
                                       ▼
┌─────────────────────────────────────────────────────────────────────────────┐
│                              DATABASE LAYER                                  │
│                              (PostgreSQL)                                    │
├─────────────────────────────────────────────────────────────────────────────┤
│                                                                              │
│  ┌──────────────┐  ┌──────────────┐  ┌──────────────┐  ┌──────────────┐    │
│  │   company    │  │  authority   │  │   sub_type   │  │  documents   │    │
│  │              │  │              │  │              │  │              │    │
│  │ • id         │  │ • id         │  │ • id         │  │ • id         │    │
│  │ • company_   │  │ • authority_ │  │ • type_name  │  │ • document_  │    │
│  │   name       │  │   name       │  │ • frequency  │  │   name       │    │
│  │ • fiscal_    │  │ • status     │  │ • frequency_ │  │ • status     │    │
│  │   year       │  │              │  │   start_date │  │              │    │
│  │ • status     │  │              │  │ • due_in_    │  │              │    │
│  │              │  │              │  │   same_next_ │  │              │    │
│  │              │  │              │  │   month      │  │              │    │
│  │              │  │              │  │ • document_  │  │              │    │
│  │              │  │              │  │   start_date │  │              │    │
│  └──────┬───────┘  └──────┬───────┘  └──────┬───────┘  └──────┬───────┘    │
│         │                 │                 │                 │             │
│         │                 │                 │                 │             │
│         └─────────────────┴────────┬────────┴─────────────────┘             │
│                                    │                                         │
│  ┌─────────────────────────────────▼─────────────────────────────────────┐  │
│  │                      uploaded_documents                                │  │
│  │  • id              • document_id        • document_month              │  │
│  │  • company_id      • type_id            • document_year               │  │
│  │  • authority_id    • file_name          • is_deleted                  │  │
│  │  • uploaded_at     • uploaded_by                                      │  │
│  └───────────────────────────────────────────────────────────────────────┘  │
│                                    │                                         │
│  ┌─────────────────────────────────▼─────────────────────────────────────┐  │
│  │                         mandatory_documents                            │  │
│  │  • id              • type_id            • document_id                 │  │
│  └───────────────────────────────────────────────────────────────────────┘  │
│                                    │                                         │
│  ┌─────────────────────────────────▼─────────────────────────────────────┐  │
│  │                            favourites                                  │  │
│  │  • id              • user_id            • upload_id                   │  │
│  │  • created_at                                                         │  │
│  └───────────────────────────────────────────────────────────────────────┘  │
│                                                                              │
└─────────────────────────────────────────────────────────────────────────────┘
```

### Data Flow Diagram

```
┌─────────────┐     ┌──────────────┐     ┌─────────────────┐     ┌──────────────┐
│   User      │────▶│  Dashboard   │────▶│ Dashboard_model │────▶│  PostgreSQL  │
│  Browser    │     │  Controller  │     │                 │     │   Database   │
└─────────────┘     └──────────────┘     └─────────────────┘     └──────────────┘
      │                    │                      │                      │
      │  1. Request        │                      │                      │
      │  Dashboard         │                      │                      │
      │───────────────────▶│                      │                      │
      │                    │  2. Call Model       │                      │
      │                    │  Methods             │                      │
      │                    │─────────────────────▶│                      │
      │                    │                      │  3. Execute SQL      │
      │                    │                      │─────────────────────▶│
      │                    │                      │                      │
      │                    │                      │  4. Return Results   │
      │                    │                      │◀─────────────────────│
      │                    │  5. Return Data      │                      │
      │                    │◀─────────────────────│                      │
      │  6. Render View    │                      │                      │
      │  with Data         │                      │                      │
      │◀───────────────────│                      │                      │
      │                    │                      │                      │
      │  7. Click Upload   │                      │                      │
      │───────────────────▶│                      │                      │
      │                    │                      │                      │
      │  8. Redirect to    │                      │                      │
      │  Upload Page with  │                      │                      │
      │  Query Params      │                      │                      │
      │◀───────────────────│                      │                      │
```

---

## Database Schema

### Entity Relationship Diagram

```
                    ┌─────────────────┐
                    │    authority    │
                    ├─────────────────┤
                    │ id (PK)         │
                    │ authority_name  │
                    │ status          │
                    └────────┬────────┘
                             │
                             │ 1:N
                             ▼
┌─────────────────┐  ┌─────────────────┐  ┌─────────────────┐
│    company      │  │    sub_type     │  │   documents     │
├─────────────────┤  ├─────────────────┤  ├─────────────────┤
│ id (PK)         │  │ id (PK)         │  │ id (PK)         │
│ company_name    │  │ authority_id(FK)│  │ document_name   │
│ fiscal_year     │  │ type_name       │  │ status          │
│ status          │  │ frequency       │  └────────┬────────┘
└────────┬────────┘  │ frequency_start │           │
         │           │   _date         │           │
         │           │ due_in_same_    │           │
         │           │   next_month    │           │
         │           │ document_start_ │           │
         │           │   date          │           │
         │           │ document_name   │           │
         │           │ status          │           │
         │           └────────┬────────┘           │
         │                    │                    │
         │                    │                    │
         │        ┌───────────┴───────────┐        │
         │        │                       │        │
         │        ▼                       ▼        │
         │  ┌─────────────────┐  ┌─────────────────┐
         │  │mandatory_       │  │                 │
         │  │documents        │  │                 │
         │  ├─────────────────┤  │                 │
         │  │ id (PK)         │◀─┘                 │
         │  │ type_id (FK)    │                    │
         │  │ document_id(FK) │◀───────────────────┘
         │  └─────────────────┘
         │
         │           ┌─────────────────────────────┐
         │           │     uploaded_documents      │
         │           ├─────────────────────────────┤
         └──────────▶│ id (PK)                     │
                     │ company_id (FK)             │
                     │ authority_id (FK)           │
                     │ type_id (FK)                │
                     │ document_id (FK)            │
                     │ document_month              │
                     │ document_year               │
                     │ file_name                   │
                     │ uploaded_at                 │
                     │ uploaded_by                 │
                     │ is_deleted                  │
                     └──────────────┬──────────────┘
                                    │
                                    │ 1:N
                                    ▼
                     ┌─────────────────────────────┐
                     │        favourites           │
                     ├─────────────────────────────┤
                     │ id (PK)                     │
                     │ user_id (FK)                │
                     │ upload_id (FK)              │
                     │ created_at                  │
                     └─────────────────────────────┘
```

### Key Tables Description

| Table | Purpose |
|-------|---------|
| `company` | Stores company information including fiscal year settings |
| `authority` | Regulatory authorities (e.g., GST, PF, ESI) |
| `sub_type` | Document types with frequency and due date configurations |
| `documents` | Master list of document names |
| `mandatory_documents` | Links document types to required documents |
| `uploaded_documents` | Tracks all uploaded documents with metadata |
| `favourites` | User's favourite documents for quick access |

### Key Columns in sub_type

| Column | Description | Values |
|--------|-------------|--------|
| `frequency` | Upload frequency | Monthly, Quarterly, Half Yearly, Yearly, One Time |
| `frequency_start_date` | Day of month when due | 1-31 |
| `due_in_same_next_month` | When document is due | 0 = Same month, 1 = Next month |
| `document_start_date` | When tracking begins | Date format |
| `document_name` | Ordered list of document IDs | Comma-separated IDs |

---

## Component Details

### Dashboard Controller (`application/controllers/dashboard.php`)

```php
class Dashboard extends CI_Controller {
    
    public function __construct() {
        parent::__construct();
        $this->load->model('Dashboard_model');
    }
    
    // Main dashboard view
    public function index() { ... }
    
    // AJAX: Toggle favourite status
    public function toggleFavourite() { ... }
    
    // AJAX: Get tab-specific data
    public function getTabData() { ... }
    
    // AJAX: Get updated counts
    public function getCounts() { ... }
}
```

### Dashboard Model (`application/models/Dashboard_model.php`)

| Method | Purpose |
|--------|---------|
| `getFavourites($user_id)` | Get user's favourite documents |
| `getPendingDocuments()` | Get documents due this month (not yet overdue) |
| `getOverdueDocuments()` | Get past-due documents |
| `getUpcomingDocuments()` | Get documents due next month |
| `getUploadedDocuments()` | Get all uploaded documents |
| `getDashboardCounts($user_id)` | Get counts for all tabs |
| `toggleFavourite($user_id, $upload_id)` | Add/remove favourite |
| `_getPreviousMonthsOverdue()` | Get overdue from previous months |

### Dashboard View (`application/views/dashboard.php`)

**Tab Structure:**
```html
┌─────────────────────────────────────────────────────────────┐
│  [★ Favourite] [⏳ Pending] [⚠ Overdue] [📅 Upcoming] [✓ Uploaded] │
├─────────────────────────────────────────────────────────────┤
│                                                              │
│  ┌─────────┐ ┌─────────┐ ┌─────────┐ ┌─────────┐ ┌─────────┐ │
│  │ Count   │ │ Count   │ │ Count   │ │ Count   │ │ Count   │ │
│  │  Card   │ │  Card   │ │  Card   │ │  Card   │ │  Card   │ │
│  └─────────┘ └─────────┘ └─────────┘ └─────────┘ └─────────┘ │
│                                                              │
│  ┌──────────────────────────────────────────────────────────┐│
│  │                     Data Table                           ││
│  │  Company | Authority | Type | Documents | Period | Action││
│  └──────────────────────────────────────────────────────────┘│
└─────────────────────────────────────────────────────────────┘
```

### Upload View Integration

When clicking "Upload" from dashboard:
```
Dashboard Row → Click Upload → Redirect to:
/upload/index/{company_id}?authority_id=X&type_id=Y&month=M&year=YYYY
```

**Auto-populated fields:**
- Year dropdown
- Month dropdown  
- Authority dropdown
- Document Head (Type) dropdown
- Document list loaded automatically

---

## Business Logic

### Document Status Determination

```
┌─────────────────────────────────────────────────────────────┐
│                    PENDING DOCUMENTS                         │
├─────────────────────────────────────────────────────────────┤
│ Criteria:                                                    │
│ • Current month documents                                    │
│ • Due day >= today's date                                   │
│ • Not yet uploaded                                          │
│ • Document start date <= current date                       │
│                                                              │
│ If due_in_same_next_month = 1 (next month):                 │
│   → Document is for PREVIOUS month                          │
│ If due_in_same_next_month = 0 (same month):                 │
│   → Document is for CURRENT month                           │
└─────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────┐
│                    OVERDUE DOCUMENTS                         │
├─────────────────────────────────────────────────────────────┤
│ Criteria:                                                    │
│ • Due day < today's date (for current month)                │
│ • OR any previous month with missing documents              │
│ • Not uploaded                                              │
│ • Document start date <= current date                       │
└─────────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────────┐
│                   UPCOMING DOCUMENTS                         │
├─────────────────────────────────────────────────────────────┤
│ Criteria:                                                    │
│ • Next month documents                                       │
│ • Document start date <= next month                         │
│ • Considers frequency (Monthly, Quarterly, etc.)            │
└─────────────────────────────────────────────────────────────┘
```

### Frequency Calculation

```
Monthly:     Due every month
Quarterly:   Due every 3 months (based on fiscal year)
Half Yearly: Due every 6 months (based on fiscal year)
Yearly:      Due once per year (based on fiscal year)
One Time:    Due once only
```

### Fiscal Year Support

```
Company fiscal_year examples:
• "jan-dec" → Q1: Jan-Mar, Q2: Apr-Jun, Q3: Jul-Sep, Q4: Oct-Dec
• "apr-mar" → Q1: Apr-Jun, Q2: Jul-Sep, Q3: Oct-Dec, Q4: Jan-Mar
```

---

## API Endpoints

### Dashboard Endpoints

| Endpoint | Method | Purpose |
|----------|--------|---------|
| `/dashboard` | GET | Load main dashboard |
| `/dashboard/toggleFavourite` | POST | Toggle favourite status |
| `/dashboard/getTabData` | POST | Get data for specific tab |
| `/dashboard/getCounts` | POST | Get updated tab counts |

### Upload Endpoints (Used by Dashboard)

| Endpoint | Method | Purpose |
|----------|--------|---------|
| `/upload/index/{company_id}` | GET | Load upload page |
| `/upload/getTypes` | POST | Get document types for authority |
| `/upload/getDocuments` | POST | Get documents for type |
| `/upload/upload` | POST | Upload document file |

### Query Parameters for Upload (from Dashboard)

| Parameter | Description |
|-----------|-------------|
| `authority_id` | Pre-select authority dropdown |
| `type_id` | Pre-select document head dropdown |
| `month` | Pre-select month dropdown |
| `year` | Pre-select year dropdown |

---

## Runbook

### 1. Deployment Checklist

```bash
# Pre-deployment
□ Backup database
□ Backup existing files
□ Verify PostgreSQL connection

# Files to deploy
□ application/controllers/dashboard.php
□ application/models/Dashboard_model.php
□ application/views/dashboard.php
□ application/views/company/upload_view.php (updated)
□ application/views/templates/header.php (if modified)

# Post-deployment
□ Clear browser cache
□ Verify favourites table exists
□ Test all 5 tabs
□ Test upload from dashboard
□ Verify existing upload functionality works
```

### 2. Database Setup

```sql
-- Create favourites table (auto-created by model if missing)
CREATE TABLE IF NOT EXISTS favourites (
    id SERIAL PRIMARY KEY,
    user_id INTEGER NOT NULL,
    upload_id INTEGER NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(user_id, upload_id)
);

-- Create index for performance
CREATE INDEX idx_favourites_user_id ON favourites(user_id);
CREATE INDEX idx_favourites_upload_id ON favourites(upload_id);
```

### 3. Configuration Verification

```php
// Verify in application/config/database.php
$db['default'] = array(
    'dsn'      => '',
    'hostname' => 'localhost',
    'username' => 'postgres',
    'password' => '***',
    'database' => 'dmstype',
    'dbdriver' => 'postgre',
    // ... other settings
);
```

### 4. Health Check Commands

```sql
-- Check document counts
SELECT 
    (SELECT COUNT(*) FROM uploaded_documents WHERE is_deleted = 0) as total_uploaded,
    (SELECT COUNT(*) FROM favourites) as total_favourites,
    (SELECT COUNT(*) FROM sub_type WHERE status = '1') as active_types,
    (SELECT COUNT(*) FROM company WHERE status = '1') as active_companies;

-- Check for overdue documents
SELECT 
    c.company_name,
    st.type_name,
    COUNT(*) as pending_docs
FROM sub_type st
CROSS JOIN company c
LEFT JOIN uploaded_documents ud ON ud.type_id = st.id AND ud.company_id = c.id
WHERE st.status = '1' AND c.status = '1' AND ud.id IS NULL
GROUP BY c.company_name, st.type_name
ORDER BY pending_docs DESC
LIMIT 10;
```

### 5. Common Operations

#### Clear User Favourites
```sql
DELETE FROM favourites WHERE user_id = ?;
```

#### Reset Document Status
```sql
-- Mark document as deleted (soft delete)
UPDATE uploaded_documents 
SET is_deleted = 1 
WHERE id = ?;
```

#### Check User Session
```php
// In controller
$user_id = $this->session->userdata('user_id');
if (!$user_id) {
    redirect('login');
}
```

---

## Troubleshooting Guide

### Issue 1: "relation 'favourites' does not exist"

**Cause:** Favourites table not created  
**Solution:** 
```sql
CREATE TABLE IF NOT EXISTS favourites (
    id SERIAL PRIMARY KEY,
    user_id INTEGER NOT NULL,
    upload_id INTEGER NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(user_id, upload_id)
);
```

### Issue 2: Document counts showing incorrect numbers

**Cause:** LEFT JOIN not properly matching uploaded documents  
**Check:**
```sql
-- Verify upload records exist
SELECT * FROM uploaded_documents 
WHERE type_id = ? AND company_id = ? 
AND document_month = ? AND document_year = ?
AND is_deleted = 0;
```

### Issue 3: Upload page not pre-filling from dashboard

**Cause:** Event handlers not registered before addRow() call  
**Solution:** Ensure addRow() is called at END of jQuery ready block

**Debug Steps:**
1. Open browser console (F12)
2. Look for: "=== AUTHORITY CHANGE HANDLER ==="
3. Check for AJAX errors
4. Verify query parameters in URL

### Issue 4: Due dates calculating incorrectly

**Check sub_type configuration:**
```sql
SELECT 
    id,
    type_name,
    frequency,
    frequency_start_date,
    due_in_same_next_month,
    document_start_date
FROM sub_type 
WHERE id = ?;
```

**Verify:**
- `frequency_start_date`: Should be 1-31 (day of month)
- `due_in_same_next_month`: 0 = same month, 1 = next month
- `document_start_date`: Must be <= current date for documents to show

### Issue 5: Duplicate entries in Overdue tab

**Cause:** Date range generation including current month twice  
**Solution:** Ensure `_getPreviousMonthsOverdue()` excludes current month

### Issue 6: Browser showing old JavaScript

**Solution:**
1. Hard refresh: Ctrl + Shift + R
2. Clear browser cache
3. Check for CDN caching
4. Append version query string to JS files

### Issue 7: AJAX calls failing silently

**Debug:**
```javascript
// Add to AJAX calls
error: function(xhr, status, error) {
    console.error("AJAX Error:", status, error);
    console.error("Response:", xhr.responseText);
}
```

---

## Performance Considerations

### Database Indexes

```sql
-- Recommended indexes
CREATE INDEX idx_uploaded_docs_type ON uploaded_documents(type_id);
CREATE INDEX idx_uploaded_docs_company ON uploaded_documents(company_id);
CREATE INDEX idx_uploaded_docs_period ON uploaded_documents(document_year, document_month);
CREATE INDEX idx_uploaded_docs_deleted ON uploaded_documents(is_deleted);
CREATE INDEX idx_subtype_status ON sub_type(status);
CREATE INDEX idx_company_status ON company(status);
```

### Query Optimization

- Use LIMIT for large result sets
- Implement pagination for Uploaded tab
- Cache dashboard counts (consider Redis)
- Use database connection pooling

---

## Security Considerations

1. **Session Validation:** Always verify user_id from session
2. **SQL Injection:** Use parameterized queries (CodeIgniter Query Builder)
3. **XSS Prevention:** Escape output in views
4. **File Upload:** Validate file types and sizes
5. **Access Control:** Verify company access permissions

---

## Version History

| Version | Date | Changes |
|---------|------|---------|
| 1.0 | Dec 20, 2025 | Initial dashboard implementation |
| 1.1 | Dec 20, 2025 | Added upload from dashboard feature |
| 1.2 | Dec 20, 2025 | Fixed event handler registration order |
| 1.3 | Dec 20, 2025 | Hide "Add Row" when from dashboard |

---

## Contact

For issues or questions, contact the development team.


