# Production Package Creation Guide

## 📦 Current Size vs Target Size

| State | Size | Notes |
|-------|------|-------|
| **Current (Development)** | ~800 MB | Includes backups, logs, test uploads, git history |
| **Target (Production)** | ~50-100 MB | Clean, optimized for deployment |
| **Space Saved** | ~700-750 MB | 90% reduction! |

---

## 🎯 What Gets Removed (and Why)

### Large Items (~180 MB saved)

| Folder/File | Size | Why Remove? |
|-------------|------|-------------|
| `backup_20260213/` | 88 MB | Development backup, not needed in production |
| `uploads/documents/*` | 65 MB | Test documents/uploads, start fresh in production |
| `output/` | 9 MB | Generated zips, can be regenerated |
| `vendor/` | 10 MB | Composer dependencies, regenerate on server (best practice) |
| `system1/` | 2 MB | Duplicate CodeIgniter system folder |
| `application/logs/` | 3 MB | Development logs, start fresh |

### Development Files (~5 MB saved)

- `application/tests/` - Test files
- `application/controllers/backup/` - Controller backups
- `.git/` - Git history (can be 50-100 MB+)
- `.cursor/`, `.vscode/` - IDE files
- `*.bat`, `*.ps1` - Development scripts
- Development documentation

### SQL Scripts (Keep only production ones)

**Remove:**
- `*seed*.sql` - Test data
- `*test*.sql` - Test scripts
- `clear_*.sql`, `truncate_*.sql` - Dev cleanup scripts
- `migrate_to_multi_company.sql` - Already migrated

**Keep:**
- `production_deployment_2026_02_15.sql` ✅
- `rollback_2026_02_15.sql` ✅
- Core migration scripts ✅

---

## 🚀 Quick Start

### Method 1: Using Batch File (Easiest)

1. Double-click `CREATE_PRODUCTION_PACKAGE.bat`
2. Wait for completion (~2-5 minutes)
3. Find `dmsnew_production_v2.2.0.zip` in parent directory

### Method 2: Using PowerShell Script (More Control)

```powershell
# Basic usage (recommended)
.\create_production_package.ps1

# Keep vendor folder (if server doesn't have Composer)
.\create_production_package.ps1 -keepVendor

# Custom output location
.\create_production_package.ps1 -outputPath "D:\deployments\dmsnew_prod.zip"
```

### Method 3: Manual (For Custom Control)

See detailed manual steps at the bottom of this guide.

---

## 📋 What the Script Does

1. ✅ Creates temporary working directory
2. ✅ Copies files, excluding development items
3. ✅ Creates required empty directories
4. ✅ Adds security `index.html` files
5. ✅ Compresses to optimized ZIP
6. ✅ Shows size comparison and summary
7. ✅ Cleans up temporary files

---

## 📤 After Creating Package

### Transfer to Production Server

```bash
# Option 1: SCP (Linux/Mac)
scp dmsnew_production_v2.2.0.zip user@server:/var/www/

# Option 2: SFTP
sftp user@server
put dmsnew_production_v2.2.0.zip
exit

# Option 3: Upload via hosting control panel (cPanel, Plesk, etc.)
```

### Extract on Server

```bash
# Navigate to web root
cd /var/www/html/

# Extract ZIP
unzip dmsnew_production_v2.2.0.zip -d dmsnew/

# Or if extracted to a subfolder, move files
mv dmsnew_production_v2.2.0/* dmsnew/
```

---

## ⚙️ Server Setup (After Extraction)

### 1. Install Composer Dependencies

```bash
cd /var/www/html/dmsnew/
composer install --no-dev --optimize-autoloader
```

**Why?** The vendor folder is excluded from the package for security and size optimization. This regenerates only production dependencies.

### 2. Set File Permissions

```bash
# Folders: 755 (owner can write, others can read/execute)
find . -type d -exec chmod 755 {} \;

# Files: 644 (owner can write, others can read)
find . -type f -exec chmod 644 {} \;

# Make writable folders
chmod 777 application/cache
chmod 777 application/logs
chmod 777 uploads
chmod 777 uploads/documents
chmod 777 uploads/temp
chmod 777 uploads/authority_files
chmod 777 uploads/company
chmod 777 output
```

### 3. Update Database Configuration

```bash
nano application/config/database.php
```

Update with production credentials:

```php
'hostname' => 'your_prod_db_host',
'username' => 'your_prod_db_user',
'password' => 'your_prod_db_password',
'database' => 'your_prod_db_name',
```

### 4. Run Database Migration

```bash
psql -U your_db_user -d your_db_name -f application/sql/production_deployment_2026_02_15.sql
```

See `DEPLOYMENT_GUIDE_2026_02_15.md` for full database migration instructions.

### 5. Configure PHP (if needed)

```bash
# Upload limits
upload_max_filesize = 50M
post_max_size = 50M

# Session settings
session.save_path = "/var/lib/php/sessions"

# Error reporting (production)
display_errors = Off
log_errors = On
error_log = /var/log/php/error.log
```

### 6. Test Application

1. Visit your domain: `https://your-domain.com/dmsnew/`
2. Login as Super Admin
3. Test key features:
   - Dashboard loads
   - Can add/edit users
   - Can upload documents
   - Per-company permissions work
   - Reports generate

---

## 🔍 Verification Checklist

After deployment, verify:

- [ ] Homepage loads without errors
- [ ] Login works (Super Admin, Admin, User)
- [ ] Dashboard shows correct data
- [ ] User list filtered correctly by role
- [ ] Can create user with per-company permissions
- [ ] Can edit existing users
- [ ] Document upload works
- [ ] Document delete respects permissions
- [ ] Reports generate correctly
- [ ] No PHP errors in logs
- [ ] Database queries execute fast

---

## 📊 Package Contents

### Files Included

```
dmsnew_production/
├── application/
│   ├── config/          (database.php, config.php, etc.)
│   ├── controllers/     (all except backup/)
│   ├── models/          (all)
│   ├── views/           (all)
│   ├── helpers/         (all)
│   ├── libraries/       (all)
│   ├── third_party/     (PHPExcel, etc.)
│   ├── sql/             (production scripts only)
│   ├── cache/           (empty, with index.html)
│   └── logs/            (empty, with index.html)
├── public/
│   └── assets/          (CSS, JS, images, fonts)
├── system/              (CodeIgniter core)
├── uploads/             (empty folders with index.html)
├── index.php
├── composer.json
├── DEPLOYMENT_GUIDE_2026_02_15.md
└── readme.rst
```

### Files Excluded

- ❌ `backup_20260213/` - Development backup
- ❌ `uploads/documents/*` - Test uploads
- ❌ `output/*` - Generated zips
- ❌ `vendor/` - Regenerate with Composer
- ❌ `application/logs/log-*.php` - Old logs
- ❌ `application/tests/` - Test files
- ❌ `.git/` - Git history
- ❌ `*.bat`, `*.ps1` - Dev scripts
- ❌ Development documentation
- ❌ SQL test/seed scripts

---

## 🆘 Troubleshooting

### Issue: Package still > 200 MB

**Cause:** `.git` folder might be large  
**Solution:** Check git folder size:
```powershell
Get-ChildItem -Path .git -Recurse -File | Measure-Object -Property Length -Sum
```
If large, ensure script excludes it (it should by default).

### Issue: Vendor folder missing on server

**Cause:** Normal - it's excluded by design  
**Solution:** Run `composer install --no-dev` on server

### Issue: "Permission denied" errors on server

**Cause:** Incorrect file permissions  
**Solution:** Run the chmod commands in step 2 above

### Issue: Application not loading

**Cause:** Various possible causes  
**Solutions:**
1. Check web server error logs
2. Verify database.php configuration
3. Ensure mod_rewrite enabled (Apache)
4. Check .htaccess exists and is correct

### Issue: Database connection failed

**Cause:** Wrong credentials or database not created  
**Solution:**
1. Verify database credentials in `application/config/database.php`
2. Ensure database exists: `createdb your_db_name`
3. Run migration script

---

## 🔧 Manual Package Creation (Alternative)

If scripts don't work, create package manually:

### Step 1: Copy Core Files

```bash
# Create clean directory
mkdir dmsnew_production

# Copy essential folders
xcopy /E /I application dmsnew_production\application
xcopy /E /I public dmsnew_production\public
xcopy /E /I system dmsnew_production\system

# Copy root files
copy index.php dmsnew_production\
copy composer.json dmsnew_production\
copy readme.rst dmsnew_production\
copy DEPLOYMENT_GUIDE_2026_02_15.md dmsnew_production\
```

### Step 2: Remove Unwanted Items

```bash
cd dmsnew_production

# Remove test files
rmdir /S application\tests

# Remove logs
del application\logs\log-*.php

# Remove backups
rmdir /S application\controllers\backup
del application\config\*_orig.php
```

### Step 3: Create Empty Folders

```bash
# Create required directories
mkdir application\cache
mkdir application\logs
mkdir uploads\documents
mkdir uploads\temp
mkdir output

# Create security files
echo "" > application\cache\index.html
echo "" > application\logs\index.html
echo "" > uploads\documents\index.html
```

### Step 4: Compress

Use any compression tool (7-Zip, WinRAR, Windows built-in) to create ZIP.

---

## 📝 Notes

1. **Vendor Folder**: Best practice is to exclude and regenerate on server. Keeps package small and ensures proper autoloader optimization.

2. **Uploads Folder**: Start fresh in production. Test documents shouldn't go to production.

3. **Database**: Always backup before running migrations in production!

4. **Logs**: Start fresh in production. Old development logs contain sensitive info.

5. **Composer**: If server doesn't have Composer, use `-keepVendor` flag when creating package.

---

## 📞 Support

For issues during package creation or deployment:

1. Check this guide's troubleshooting section
2. Review `DEPLOYMENT_GUIDE_2026_02_15.md`
3. Check application logs after deployment
4. Contact development team with:
   - Package creation log
   - Server environment details
   - Error messages

---

**Version:** 2.2.0  
**Last Updated:** February 15, 2026  
**Package Script Version:** 1.0

