# Multi-Company Multi-Role Database Design

**Version:** 1.0  
**Date:** 2026-01-15  
**Backup Name:** `pre-multicompany-v1`

---

## Overview

This document describes the database design for implementing:
- Multi-company support with tenant isolation
- Role-based access control (Super Admin, Admin, User)
- Activation workflow for document heads
- Company-specific configurations

---

## Design Philosophy

- **Shared Masters** - Keep existing master tables (`authority`, `sub_type`, `documents`) as templates
- **Company-Specific Overrides** - Mapping tables for company-specific configurations
- **Activation Workflow** - All items copied as DISABLED, Admin must review and activate
- **Cascade Enablement** - When document head is activated, parent authority and child documents auto-enable

---

## Existing Tables (UNCHANGED)

These tables remain as **master templates**:

| Table | Purpose |
|-------|---------|
| `authority` | Master list of authorities (GST, PF, ESI, etc.) |
| `sub_type` | Master document heads |
| `documents` | Master document names |
| `mandatory_documents` | Default document-to-type links |
| `sub_type_reminders` | Default reminder settings |
| `document_groups` | Default document groupings |
| `role` | User roles (1=Super Admin, 2=Admin, 3=User) |
| `users` | User accounts |
| `company` | Company master |

---

## New Tables (12 Total)

### 1. company_templates

Pre-defined templates for quick company setup.

```sql
CREATE TABLE company_templates (
    id SERIAL PRIMARY KEY,
    template_name VARCHAR(100) NOT NULL,
    description TEXT,
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    created_by INT REFERENCES users(id)
);
```

### 2. company_template_document_heads

Document heads included in each template.

```sql
CREATE TABLE company_template_document_heads (
    id SERIAL PRIMARY KEY,
    template_id INT NOT NULL REFERENCES company_templates(id) ON DELETE CASCADE,
    type_id INT NOT NULL REFERENCES sub_type(id) ON DELETE CASCADE,
    UNIQUE(template_id, type_id)
);
```

### 3. company_authorities

Maps which authorities are available for each company.

```sql
CREATE TABLE company_authorities (
    id SERIAL PRIMARY KEY,
    company_id INT NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    authority_id INT NOT NULL REFERENCES authority(id) ON DELETE CASCADE,
    
    is_enabled BOOLEAN DEFAULT FALSE,      -- DISABLED by default
    enabled_at TIMESTAMP,
    enabled_by INT REFERENCES users(id),
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(company_id, authority_id)
);

CREATE INDEX idx_ca_company ON company_authorities(company_id);
CREATE INDEX idx_ca_authority ON company_authorities(authority_id);
```

### 4. company_document_heads

Company-specific document head settings with activation status.

```sql
CREATE TABLE company_document_heads (
    id SERIAL PRIMARY KEY,
    company_id INT NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    type_id INT NOT NULL REFERENCES sub_type(id) ON DELETE CASCADE,
    
    -- Status fields
    is_enabled BOOLEAN DEFAULT FALSE,       -- DISABLED by default
    is_reviewed BOOLEAN DEFAULT FALSE,
    
    -- Company-specific overrides
    custom_start_date DATE,                 -- Must be >= company.co_start_date
    custom_frequency_start_date VARCHAR(10),
    custom_due_in_same_next_month VARCHAR(1),
    custom_document_order TEXT,             -- CSV for ordering
    
    -- Audit fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    enabled_at TIMESTAMP,
    enabled_by INT REFERENCES users(id),
    created_by INT REFERENCES users(id),
    
    UNIQUE(company_id, type_id)
);

CREATE INDEX idx_cdh_company ON company_document_heads(company_id);
CREATE INDEX idx_cdh_type ON company_document_heads(type_id);
```

### 5. company_mandatory_documents

Company-specific mandatory document settings with ordering.

```sql
CREATE TABLE company_mandatory_documents (
    id SERIAL PRIMARY KEY,
    company_id INT NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    type_id INT NOT NULL REFERENCES sub_type(id) ON DELETE CASCADE,
    document_id INT NOT NULL REFERENCES documents(id) ON DELETE CASCADE,
    
    is_mandatory BOOLEAN DEFAULT FALSE,
    is_enabled BOOLEAN DEFAULT FALSE,       -- Follows parent doc head
    sort_order INT DEFAULT 0,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    UNIQUE(company_id, type_id, document_id)
);

CREATE INDEX idx_cmd_company ON company_mandatory_documents(company_id);
CREATE INDEX idx_cmd_type ON company_mandatory_documents(type_id);
```

### 6. company_document_head_reminders

Company-specific reminder settings.

```sql
CREATE TABLE company_document_head_reminders (
    id SERIAL PRIMARY KEY,
    company_id INT NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    type_id INT NOT NULL REFERENCES sub_type(id) ON DELETE CASCADE,
    
    reminder_no INT NOT NULL CHECK (reminder_no BETWEEN 1 AND 5),
    days_before INT NOT NULL,
    reminder_to_user BOOLEAN DEFAULT FALSE,
    reminder_to_admin BOOLEAN DEFAULT FALSE,
    reminder_to_super_admin BOOLEAN DEFAULT FALSE,
    
    is_enabled BOOLEAN DEFAULT FALSE,       -- Follows parent doc head
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    UNIQUE(company_id, type_id, reminder_no)
);

CREATE INDEX idx_cdhr_company ON company_document_head_reminders(company_id);
CREATE INDEX idx_cdhr_type ON company_document_head_reminders(type_id);
```

### 7. company_document_groups

Company-specific document groupings.

```sql
CREATE TABLE company_document_groups (
    id SERIAL PRIMARY KEY,
    company_id INT NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    parent_document_id INT NOT NULL REFERENCES documents(id) ON DELETE CASCADE,
    child_document_ids TEXT NOT NULL,       -- CSV of child document IDs
    
    is_enabled BOOLEAN DEFAULT FALSE,       -- Follows related doc head
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    UNIQUE(company_id, parent_document_id)
);

CREATE INDEX idx_cdg_company ON company_document_groups(company_id);
CREATE INDEX idx_cdg_parent ON company_document_groups(parent_document_id);
```

### 8. company_documents

Maps which documents are available for each company.

```sql
CREATE TABLE company_documents (
    id SERIAL PRIMARY KEY,
    company_id INT NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    document_id INT NOT NULL REFERENCES documents(id) ON DELETE CASCADE,
    
    is_enabled BOOLEAN DEFAULT FALSE,       -- Auto-enabled when parent doc head enabled
    enabled_at TIMESTAMP,
    
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(company_id, document_id)
);

CREATE INDEX idx_cd_company ON company_documents(company_id);
CREATE INDEX idx_cd_document ON company_documents(document_id);
```

### 9. user_companies

Assigns users to specific companies.

```sql
CREATE TABLE user_companies (
    id SERIAL PRIMARY KEY,
    user_id INT NOT NULL REFERENCES users(id) ON DELETE CASCADE,
    company_id INT NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(user_id, company_id)
);

CREATE INDEX idx_uc_user ON user_companies(user_id);
CREATE INDEX idx_uc_company ON user_companies(company_id);
```

### 10. user_authorities

Assigns users to specific authorities (granular access for User role).

```sql
CREATE TABLE user_authorities (
    id SERIAL PRIMARY KEY,
    user_id INT NOT NULL REFERENCES users(id) ON DELETE CASCADE,
    authority_id INT NOT NULL REFERENCES authority(id) ON DELETE CASCADE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(user_id, authority_id)
);

CREATE INDEX idx_ua_user ON user_authorities(user_id);
CREATE INDEX idx_ua_authority ON user_authorities(authority_id);
```

### 11. user_document_types

Assigns users to specific document types (granular access for User role).

```sql
CREATE TABLE user_document_types (
    id SERIAL PRIMARY KEY,
    user_id INT NOT NULL REFERENCES users(id) ON DELETE CASCADE,
    type_id INT NOT NULL REFERENCES sub_type(id) ON DELETE CASCADE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(user_id, type_id)
);

CREATE INDEX idx_udt_user ON user_document_types(user_id);
CREATE INDEX idx_udt_type ON user_document_types(type_id);
```

### 12. role_permissions

Defines what actions each role can perform.

```sql
CREATE TABLE role_permissions (
    id SERIAL PRIMARY KEY,
    role_id INT NOT NULL REFERENCES role(id) ON DELETE CASCADE,
    permission_key VARCHAR(100) NOT NULL,
    is_allowed BOOLEAN DEFAULT FALSE,
    UNIQUE(role_id, permission_key)
);

CREATE INDEX idx_rp_role ON role_permissions(role_id);
```

---

## Activation Workflow

### Company Creation Flow

```
1. Admin creates company with template
   ↓
2. System copies all items as DISABLED
   - company_authorities.is_enabled = FALSE
   - company_document_heads.is_enabled = FALSE
   - company_mandatory_documents.is_enabled = FALSE
   - company_document_head_reminders.is_enabled = FALSE
   - company_documents.is_enabled = FALSE
   ↓
3. Admin goes to "Company Document Heads" menu
   ↓
4. Admin reviews each document head:
   - Sets document start date (>= company start date)
   - Reviews mandatory documents
   - Reviews reminders
   ↓
5. Admin clicks "ACTIVATE"
   ↓
6. System auto-enables:
   ✓ Document Head (is_enabled = TRUE)
   ✓ Parent Authority (auto-enabled)
   ✓ Child Documents (auto-enabled)
   ✓ Reminders (auto-enabled)
   ↓
7. Now visible in: Upload, View, Reports, Compliance
```

### Activation Cascade

When a document head is activated:

| Component | Action |
|-----------|--------|
| Document Head | Set `is_enabled = TRUE` |
| Parent Authority | Auto-enable in `company_authorities` |
| Mandatory Documents | Auto-enable in `company_mandatory_documents` |
| Documents | Auto-enable in `company_documents` |
| Reminders | Auto-enable in `company_document_head_reminders` |
| Document Groups | Auto-enable if parent document is in this doc head |

---

## Validation Rules

### Document Start Date Validation

```
RULE: custom_start_date >= company.co_start_date

Example:
  Company Start: 01-Apr-2024
  ✓ Doc Start: 01-Apr-2024 (OK - same as company)
  ✓ Doc Start: 15-May-2024 (OK - after company start)
  ✗ Doc Start: 01-Jan-2024 (ERROR - before company start)
```

---

## Access Control Matrix

| Role | Companies | Doc Heads | Custom Start Date | Mandatory Docs | Reminders |
|------|-----------|-----------|-------------------|----------------|-----------|
| Super Admin | All | Full CRUD | Yes | Full CRUD | Full CRUD |
| Admin | Assigned | Add/Edit/Activate | Yes (with validation) | Edit | Edit |
| User | Assigned | View only | No | View only | View only |

---

## Permission Keys

| Key | Super Admin | Admin | User |
|-----|:-----------:|:-----:|:----:|
| company.create | Yes | No | No |
| company.edit | Yes | Yes | No |
| company.delete | Yes | No | No |
| company.view | Yes | Yes | Yes |
| company_doc_heads.view | Yes | Yes | Yes |
| company_doc_heads.add | Yes | Yes | No |
| company_doc_heads.edit | Yes | Yes | No |
| company_doc_heads.activate | Yes | Yes | No |
| company_doc_heads.delete | Yes | Yes | No |
| template.manage | Yes | No | No |
| authority.manage | Yes | Yes | No |
| document_head.manage | Yes | Yes | No |
| document.manage | Yes | Yes | No |
| user.manage | Yes | Yes | No |
| role.manage | Yes | No | No |
| holiday.manage | Yes | Yes | No |
| upload.create | Yes | Yes | No |
| upload.edit | Yes | Yes | No |
| upload.delete | Yes | Yes | No |
| upload.view | Yes | Yes | Yes |
| dashboard.view | Yes | Yes | Yes |

---

## UI: Company Document Heads Menu

### Location
Masters → Company Document Heads

### List View

| Document Head | Authority | Status | Start Date | Actions |
|---------------|-----------|--------|------------|---------|
| GST Monthly | GST | 🔴 Pending | - | [Review] |
| GST Quarterly | GST | 🟡 Reviewed | 01-Apr-2024 | [Activate] |
| PF Monthly | PF | 🟢 Active | 15-Apr-2024 | [Edit] [Deactivate] |

**Status Legend:**
- 🔴 Pending Review - `is_reviewed = FALSE, is_enabled = FALSE`
- 🟡 Reviewed - `is_reviewed = TRUE, is_enabled = FALSE`
- 🟢 Active - `is_enabled = TRUE`

---

## Migration Scripts

### Create All Tables

Run the SQL statements from sections 1-12 above.

### Initialize Company Data from Template

```sql
-- When company is created with template_id

-- 1. Copy authorities as DISABLED
INSERT INTO company_authorities (company_id, authority_id, is_enabled)
SELECT DISTINCT
    :new_company_id,
    CAST(st.authority_id AS INT),
    FALSE
FROM company_template_document_heads ctdh
JOIN sub_type st ON st.id = ctdh.type_id
WHERE ctdh.template_id = :template_id;

-- 2. Copy document heads as DISABLED
INSERT INTO company_document_heads 
    (company_id, type_id, is_enabled, is_reviewed, created_by)
SELECT 
    :new_company_id,
    ctdh.type_id,
    FALSE,
    FALSE,
    :admin_id
FROM company_template_document_heads ctdh
WHERE ctdh.template_id = :template_id;

-- 3. Copy mandatory documents as DISABLED
INSERT INTO company_mandatory_documents 
    (company_id, type_id, document_id, is_mandatory, is_enabled, sort_order)
SELECT 
    :new_company_id,
    md.type_id,
    md.document_id,
    CASE WHEN md.mandatory = '1' THEN TRUE ELSE FALSE END,
    FALSE,
    ROW_NUMBER() OVER (PARTITION BY md.type_id ORDER BY md.id)
FROM mandatory_documents md
JOIN company_template_document_heads ctdh ON md.type_id = ctdh.type_id
WHERE ctdh.template_id = :template_id;

-- 4. Copy reminders as DISABLED
INSERT INTO company_document_head_reminders 
    (company_id, type_id, reminder_no, days_before, 
     reminder_to_user, reminder_to_admin, reminder_to_super_admin, is_enabled)
SELECT 
    :new_company_id,
    sr.sub_type_id,
    sr.reminder_no,
    sr.days_before,
    sr.reminder_to_user = 'true',
    sr.reminder_to_admin = 'true',
    sr.reminder_to_super_admin = 'true',
    FALSE
FROM sub_type_reminders sr
JOIN company_template_document_heads ctdh ON sr.sub_type_id = ctdh.type_id
WHERE ctdh.template_id = :template_id;

-- 5. Copy documents as DISABLED
INSERT INTO company_documents (company_id, document_id, is_enabled)
SELECT DISTINCT
    :new_company_id,
    md.document_id,
    FALSE
FROM mandatory_documents md
JOIN company_template_document_heads ctdh ON md.type_id = ctdh.type_id
WHERE ctdh.template_id = :template_id;

-- 6. Copy document groups as DISABLED
INSERT INTO company_document_groups (company_id, parent_document_id, child_document_ids, is_enabled)
SELECT 
    :new_company_id,
    dg.document_id,
    dg.grouped_doc_id,
    FALSE
FROM document_groups dg
WHERE dg.document_id IN (
    SELECT DISTINCT md.document_id
    FROM mandatory_documents md
    JOIN company_template_document_heads ctdh ON md.type_id = ctdh.type_id
    WHERE ctdh.template_id = :template_id
);
```

### Migrate Existing User CSV Data

```sql
-- Migrate users.user_company to user_companies
INSERT INTO user_companies (user_id, company_id)
SELECT u.id, unnest(string_to_array(u.user_company, ','))::int
FROM users u
WHERE u.user_company IS NOT NULL AND u.user_company != ''
ON CONFLICT DO NOTHING;

-- Migrate users.user_type to user_document_types
INSERT INTO user_document_types (user_id, type_id)
SELECT u.id, unnest(string_to_array(u.user_type, ','))::int
FROM users u
WHERE u.user_type IS NOT NULL AND u.user_type != ''
ON CONFLICT DO NOTHING;
```

### Seed Default Role Permissions

```sql
INSERT INTO role_permissions (role_id, permission_key, is_allowed) VALUES
-- Super Admin (1)
(1, 'company.create', TRUE), (1, 'company.edit', TRUE), (1, 'company.delete', TRUE),
(1, 'company.view', TRUE), (1, 'company_doc_heads.view', TRUE), (1, 'company_doc_heads.add', TRUE),
(1, 'company_doc_heads.edit', TRUE), (1, 'company_doc_heads.activate', TRUE),
(1, 'company_doc_heads.delete', TRUE), (1, 'template.manage', TRUE),
(1, 'authority.manage', TRUE), (1, 'document_head.manage', TRUE), (1, 'document.manage', TRUE),
(1, 'user.manage', TRUE), (1, 'role.manage', TRUE), (1, 'holiday.manage', TRUE),
(1, 'upload.create', TRUE), (1, 'upload.edit', TRUE), (1, 'upload.delete', TRUE),
(1, 'upload.view', TRUE), (1, 'dashboard.view', TRUE),

-- Admin (2)
(2, 'company.create', FALSE), (2, 'company.edit', TRUE), (2, 'company.delete', FALSE),
(2, 'company.view', TRUE), (2, 'company_doc_heads.view', TRUE), (2, 'company_doc_heads.add', TRUE),
(2, 'company_doc_heads.edit', TRUE), (2, 'company_doc_heads.activate', TRUE),
(2, 'company_doc_heads.delete', TRUE), (2, 'template.manage', FALSE),
(2, 'authority.manage', TRUE), (2, 'document_head.manage', TRUE), (2, 'document.manage', TRUE),
(2, 'user.manage', TRUE), (2, 'role.manage', FALSE), (2, 'holiday.manage', TRUE),
(2, 'upload.create', TRUE), (2, 'upload.edit', TRUE), (2, 'upload.delete', TRUE),
(2, 'upload.view', TRUE), (2, 'dashboard.view', TRUE),

-- User (3)
(3, 'company.create', FALSE), (3, 'company.edit', FALSE), (3, 'company.delete', FALSE),
(3, 'company.view', TRUE), (3, 'company_doc_heads.view', TRUE), (3, 'company_doc_heads.add', FALSE),
(3, 'company_doc_heads.edit', FALSE), (3, 'company_doc_heads.activate', FALSE),
(3, 'company_doc_heads.delete', FALSE), (3, 'template.manage', FALSE),
(3, 'authority.manage', FALSE), (3, 'document_head.manage', FALSE), (3, 'document.manage', FALSE),
(3, 'user.manage', FALSE), (3, 'role.manage', FALSE), (3, 'holiday.manage', FALSE),
(3, 'upload.create', FALSE), (3, 'upload.edit', FALSE), (3, 'upload.delete', FALSE),
(3, 'upload.view', TRUE), (3, 'dashboard.view', TRUE)
ON CONFLICT (role_id, permission_key) DO UPDATE SET is_allowed = EXCLUDED.is_allowed;
```

---

## Rollback Information

**Backup Name:** `pre-multicompany-v1`  
**Date:** 2026-01-15  

### Backup Files Created

| Backup Type | File/Name | Location | Size |
|-------------|-----------|----------|------|
| Git Tag | `pre-multicompany-v1` | Git repository | - |
| ZIP Backup | `backup_pre-multicompany-v1_20260115.zip` | `C:\xampp\htdocs\dmsnew\` | ~86 MB |
| DB Schema | `backup_schema_pre-multicompany-v1.sql` | `C:\xampp\htdocs\dmsnew\` | ~28 KB |

---

### Rollback Steps

#### Option 1: Rollback Using Git Tag

```powershell
# Navigate to project directory
cd C:\xampp\htdocs\dmsnew

# View available tags
git tag -l

# Checkout the backup tag
git checkout pre-multicompany-v1

# If you want to create a new branch from this point
git checkout -b rollback-branch pre-multicompany-v1
```

#### Option 2: Rollback Using ZIP Backup

1. Stop Apache/XAMPP server
2. Rename current `application` folder to `application_backup`
3. Extract `backup_pre-multicompany-v1_20260115.zip`
4. Copy extracted folders to `C:\xampp\htdocs\dmsnew\`
5. Restart Apache/XAMPP server

```powershell
# PowerShell commands
cd C:\xampp\htdocs\dmsnew
Rename-Item -Path "application" -NewName "application_backup"
Expand-Archive -Path "backup_pre-multicompany-v1_20260115.zip" -DestinationPath "." -Force
```

---

### Database Rollback

#### Step 1: Drop New Tables (If Created)

```sql
-- Connect to PostgreSQL and run:
DROP TABLE IF EXISTS role_permissions CASCADE;
DROP TABLE IF EXISTS user_document_types CASCADE;
DROP TABLE IF EXISTS user_authorities CASCADE;
DROP TABLE IF EXISTS user_companies CASCADE;
DROP TABLE IF EXISTS company_documents CASCADE;
DROP TABLE IF EXISTS company_document_groups CASCADE;
DROP TABLE IF EXISTS company_document_head_reminders CASCADE;
DROP TABLE IF EXISTS company_mandatory_documents CASCADE;
DROP TABLE IF EXISTS company_document_heads CASCADE;
DROP TABLE IF EXISTS company_authorities CASCADE;
DROP TABLE IF EXISTS company_template_document_heads CASCADE;
DROP TABLE IF EXISTS company_templates CASCADE;
```

#### Step 2: Restore Original Schema (If Needed)

```powershell
# Using PowerShell
cd C:\xampp\htdocs\dmsnew
$env:PGPASSWORD='postgres'
& "C:\Program Files\PostgreSQL\18\bin\psql.exe" -h localhost -U postgres -d new_dms_type -f backup_schema_pre-multicompany-v1.sql
```

Or using pgAdmin:
1. Open pgAdmin
2. Connect to `localhost` → `new_dms_type` database
3. Right-click → Query Tool
4. File → Open → Select `backup_schema_pre-multicompany-v1.sql`
5. Execute (F5)

---

### Verification After Rollback

```powershell
# Check git status
cd C:\xampp\htdocs\dmsnew
git status
git log --oneline -5

# Check if application loads
# Open browser: http://localhost/dmsnew/
```

```sql
-- Verify database tables (should NOT have new tables)
SELECT table_name 
FROM information_schema.tables 
WHERE table_schema = 'public' 
AND table_name LIKE 'company_%'
ORDER BY table_name;
```

---

### Important Notes

1. **Always backup before rollback** - Create a new backup of current state before rolling back
2. **Database data** - The schema backup only contains structure, not data
3. **Uploaded files** - The `uploads/` folder is not included in ZIP backup to save space
4. **Git tag is permanent** - The `pre-multicompany-v1` tag will remain in git history

---

### Contact for Issues

If rollback fails or causes issues:
1. Check Apache error logs: `C:\xampp\apache\logs\error.log`
2. Check PHP error logs: `C:\xampp\php\logs\php_error_log`
3. Check PostgreSQL logs via pgAdmin

