-- ============================================================================
-- Migration Script: Single Company to Multi-Company DMS
-- ============================================================================
-- This script migrates an existing single-company DMS setup to multi-company
-- Run this script in PostgreSQL after backing up your database
-- 
-- IMPORTANT: Replace {COMPANY_ID} with the actual company ID you want to migrate to
-- ============================================================================

-- Set the company ID for migration (change this to your target company)
\set TARGET_COMPANY_ID 1

-- ============================================================================
-- STEP 1: Create Multi-Company Tables (if not exist)
-- ============================================================================

-- Company Authorities Table
CREATE TABLE IF NOT EXISTS company_authorities (
    id SERIAL PRIMARY KEY,
    company_id INTEGER NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    authority_id INTEGER NOT NULL REFERENCES authority(id) ON DELETE CASCADE,
    is_enabled BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(company_id, authority_id)
);

-- Company Documents Table
CREATE TABLE IF NOT EXISTS company_documents (
    id SERIAL PRIMARY KEY,
    company_id INTEGER NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    document_id INTEGER NOT NULL REFERENCES documents(id) ON DELETE CASCADE,
    is_enabled BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(company_id, document_id)
);

-- Company Document Heads Table
CREATE TABLE IF NOT EXISTS company_document_heads (
    id SERIAL PRIMARY KEY,
    company_id INTEGER NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    type_id INTEGER NOT NULL REFERENCES sub_type(id) ON DELETE CASCADE,
    is_enabled BOOLEAN DEFAULT FALSE,
    is_reviewed BOOLEAN DEFAULT FALSE,
    custom_start_date DATE,
    custom_frequency_start_date VARCHAR(10),
    custom_due_in_same_next_month VARCHAR(10),
    enabled_at TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    created_by INTEGER,
    UNIQUE(company_id, type_id)
);

-- Company Mandatory Documents Table
CREATE TABLE IF NOT EXISTS company_mandatory_documents (
    id SERIAL PRIMARY KEY,
    company_id INTEGER NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    type_id INTEGER NOT NULL REFERENCES sub_type(id) ON DELETE CASCADE,
    document_id INTEGER NOT NULL REFERENCES documents(id) ON DELETE CASCADE,
    is_mandatory BOOLEAN DEFAULT FALSE,
    is_enabled BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(company_id, type_id, document_id)
);

-- Company Document Head Reminders Table
CREATE TABLE IF NOT EXISTS company_document_head_reminders (
    id SERIAL PRIMARY KEY,
    company_id INTEGER NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    type_id INTEGER NOT NULL REFERENCES sub_type(id) ON DELETE CASCADE,
    reminder_no INTEGER NOT NULL,
    days_before INTEGER,
    reminder_to_user BOOLEAN DEFAULT FALSE,
    reminder_to_admin BOOLEAN DEFAULT FALSE,
    reminder_to_super_admin BOOLEAN DEFAULT FALSE,
    is_enabled BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(company_id, type_id, reminder_no)
);

-- Company Document Groups Table (optional, for cascade enable)
CREATE TABLE IF NOT EXISTS company_document_groups (
    id SERIAL PRIMARY KEY,
    company_id INTEGER NOT NULL REFERENCES company(id) ON DELETE CASCADE,
    document_group_id INTEGER NOT NULL,
    is_enabled BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    UNIQUE(company_id, document_group_id)
);

-- Add indexes for better performance
CREATE INDEX IF NOT EXISTS idx_company_authorities_company_id ON company_authorities(company_id);
CREATE INDEX IF NOT EXISTS idx_company_documents_company_id ON company_documents(company_id);
CREATE INDEX IF NOT EXISTS idx_company_document_heads_company_id ON company_document_heads(company_id);
CREATE INDEX IF NOT EXISTS idx_company_document_heads_type_id ON company_document_heads(type_id);
CREATE INDEX IF NOT EXISTS idx_company_mandatory_documents_company_id ON company_mandatory_documents(company_id);
CREATE INDEX IF NOT EXISTS idx_company_document_head_reminders_company_id ON company_document_head_reminders(company_id);

-- ============================================================================
-- STEP 2: Migrate Authorities to Company Authorities
-- ============================================================================

INSERT INTO company_authorities (company_id, authority_id, is_enabled, created_at)
SELECT 
    :TARGET_COMPANY_ID,
    id,
    TRUE,  -- Enable all existing authorities
    CURRENT_TIMESTAMP
FROM authority
WHERE status = '1' AND is_delete = '0'
ON CONFLICT (company_id, authority_id) DO UPDATE 
SET is_enabled = TRUE, updated_at = CURRENT_TIMESTAMP;

-- ============================================================================
-- STEP 3: Migrate Documents to Company Documents
-- ============================================================================

INSERT INTO company_documents (company_id, document_id, is_enabled, created_at)
SELECT 
    :TARGET_COMPANY_ID,
    id,
    TRUE,  -- Enable all existing documents
    CURRENT_TIMESTAMP
FROM documents
WHERE status = '1'
ON CONFLICT (company_id, document_id) DO UPDATE 
SET is_enabled = TRUE, updated_at = CURRENT_TIMESTAMP;

-- ============================================================================
-- STEP 4: Migrate Document Heads (sub_type) to Company Document Heads
-- ============================================================================

INSERT INTO company_document_heads (
    company_id, 
    type_id, 
    is_enabled, 
    is_reviewed,
    custom_start_date,
    custom_frequency_start_date,
    custom_due_in_same_next_month,
    enabled_at,
    created_at
)
SELECT 
    :TARGET_COMPANY_ID,
    id,
    TRUE,  -- Enable all existing document heads
    TRUE,  -- Mark as reviewed
    document_start_date::date,
    frequency_start_date,
    due_in_same_next_month::text,
    CURRENT_TIMESTAMP,
    CURRENT_TIMESTAMP
FROM sub_type
WHERE status = '1'
ON CONFLICT (company_id, type_id) DO UPDATE 
SET 
    is_enabled = TRUE, 
    is_reviewed = TRUE,
    updated_at = CURRENT_TIMESTAMP;

-- ============================================================================
-- STEP 5: Migrate Mandatory Documents to Company Mandatory Documents
-- ============================================================================

INSERT INTO company_mandatory_documents (
    company_id, 
    type_id, 
    document_id, 
    is_mandatory, 
    is_enabled,
    created_at
)
SELECT 
    :TARGET_COMPANY_ID,
    type_id,
    document_id,
    CASE 
        WHEN LOWER(mandatory) IN ('yes', '1', 'true', 't') THEN TRUE
        ELSE FALSE
    END,
    TRUE,
    CURRENT_TIMESTAMP
FROM mandatory_documents
ON CONFLICT (company_id, type_id, document_id) DO UPDATE 
SET 
    is_mandatory = EXCLUDED.is_mandatory,
    is_enabled = TRUE,
    updated_at = CURRENT_TIMESTAMP;

-- ============================================================================
-- STEP 6: Migrate Reminders to Company Document Head Reminders
-- ============================================================================

INSERT INTO company_document_head_reminders (
    company_id, 
    type_id, 
    reminder_no, 
    days_before, 
    reminder_to_user, 
    reminder_to_admin, 
    reminder_to_super_admin,
    is_enabled,
    created_at
)
SELECT 
    :TARGET_COMPANY_ID,
    sub_type_id,
    reminder_no,
    days_before,
    COALESCE(reminder_to_user::boolean, FALSE),
    COALESCE(reminder_to_admin::boolean, FALSE),
    COALESCE(reminder_to_super_admin::boolean, FALSE),
    TRUE,
    CURRENT_TIMESTAMP
FROM sub_type_reminders
ON CONFLICT (company_id, type_id, reminder_no) DO UPDATE 
SET 
    days_before = EXCLUDED.days_before,
    reminder_to_user = EXCLUDED.reminder_to_user,
    reminder_to_admin = EXCLUDED.reminder_to_admin,
    reminder_to_super_admin = EXCLUDED.reminder_to_super_admin,
    is_enabled = TRUE,
    updated_at = CURRENT_TIMESTAMP;

-- ============================================================================
-- STEP 7: Update Uploaded Documents with Company ID (if not already set)
-- ============================================================================

UPDATE uploaded_documents
SET company_id = :TARGET_COMPANY_ID
WHERE company_id IS NULL OR company_id = 0;

-- ============================================================================
-- STEP 8: Verify Migration Counts
-- ============================================================================

SELECT 'Migration Summary' as info;
SELECT 'company_authorities' as table_name, COUNT(*) as count FROM company_authorities WHERE company_id = :TARGET_COMPANY_ID;
SELECT 'company_documents' as table_name, COUNT(*) as count FROM company_documents WHERE company_id = :TARGET_COMPANY_ID;
SELECT 'company_document_heads' as table_name, COUNT(*) as count FROM company_document_heads WHERE company_id = :TARGET_COMPANY_ID;
SELECT 'company_mandatory_documents' as table_name, COUNT(*) as count FROM company_mandatory_documents WHERE company_id = :TARGET_COMPANY_ID;
SELECT 'company_document_head_reminders' as table_name, COUNT(*) as count FROM company_document_head_reminders WHERE company_id = :TARGET_COMPANY_ID;
SELECT 'uploaded_documents' as table_name, COUNT(*) as count FROM uploaded_documents WHERE company_id = :TARGET_COMPANY_ID;

-- ============================================================================
-- END OF MIGRATION SCRIPT
-- ============================================================================

