-- =====================================================
-- SEED DATA FOR MULTI-COMPANY SYSTEM
-- Version: 1.0
-- Date: 2026-01-15
-- =====================================================

-- =====================================================
-- SEED DEFAULT ROLE PERMISSIONS
-- Role 1 = Super Admin, Role 2 = Admin, Role 3 = User
-- =====================================================

-- Clear existing permissions (if any)
DELETE FROM role_permissions WHERE role_id IN (1, 2, 3);

-- Super Admin (Role 1) - Full Access
INSERT INTO role_permissions (role_id, permission_key, is_allowed) VALUES
(1, 'company.create', TRUE),
(1, 'company.edit', TRUE),
(1, 'company.delete', TRUE),
(1, 'company.view', TRUE),
(1, 'company_doc_heads.view', TRUE),
(1, 'company_doc_heads.add', TRUE),
(1, 'company_doc_heads.edit', TRUE),
(1, 'company_doc_heads.activate', TRUE),
(1, 'company_doc_heads.delete', TRUE),
(1, 'template.manage', TRUE),
(1, 'authority.manage', TRUE),
(1, 'document_head.manage', TRUE),
(1, 'document.manage', TRUE),
(1, 'user.manage', TRUE),
(1, 'role.manage', TRUE),
(1, 'holiday.manage', TRUE),
(1, 'upload.create', TRUE),
(1, 'upload.edit', TRUE),
(1, 'upload.delete', TRUE),
(1, 'upload.view', TRUE),
(1, 'dashboard.view', TRUE),
(1, 'report.view', TRUE),
(1, 'setting.manage', TRUE);

-- Admin (Role 2) - Limited Access (No company create/delete, no role manage)
INSERT INTO role_permissions (role_id, permission_key, is_allowed) VALUES
(2, 'company.create', FALSE),
(2, 'company.edit', TRUE),
(2, 'company.delete', FALSE),
(2, 'company.view', TRUE),
(2, 'company_doc_heads.view', TRUE),
(2, 'company_doc_heads.add', TRUE),
(2, 'company_doc_heads.edit', TRUE),
(2, 'company_doc_heads.activate', TRUE),
(2, 'company_doc_heads.delete', TRUE),
(2, 'template.manage', FALSE),
(2, 'authority.manage', TRUE),
(2, 'document_head.manage', TRUE),
(2, 'document.manage', TRUE),
(2, 'user.manage', TRUE),
(2, 'role.manage', FALSE),
(2, 'holiday.manage', TRUE),
(2, 'upload.create', TRUE),
(2, 'upload.edit', TRUE),
(2, 'upload.delete', TRUE),
(2, 'upload.view', TRUE),
(2, 'dashboard.view', TRUE),
(2, 'report.view', TRUE),
(2, 'setting.manage', FALSE);

-- User (Role 3) - View Only
INSERT INTO role_permissions (role_id, permission_key, is_allowed) VALUES
(3, 'company.create', FALSE),
(3, 'company.edit', FALSE),
(3, 'company.delete', FALSE),
(3, 'company.view', TRUE),
(3, 'company_doc_heads.view', TRUE),
(3, 'company_doc_heads.add', FALSE),
(3, 'company_doc_heads.edit', FALSE),
(3, 'company_doc_heads.activate', FALSE),
(3, 'company_doc_heads.delete', FALSE),
(3, 'template.manage', FALSE),
(3, 'authority.manage', FALSE),
(3, 'document_head.manage', FALSE),
(3, 'document.manage', FALSE),
(3, 'user.manage', FALSE),
(3, 'role.manage', FALSE),
(3, 'holiday.manage', FALSE),
(3, 'upload.create', FALSE),
(3, 'upload.edit', FALSE),
(3, 'upload.delete', FALSE),
(3, 'upload.view', TRUE),
(3, 'dashboard.view', TRUE),
(3, 'report.view', TRUE),
(3, 'setting.manage', FALSE);

-- =====================================================
-- SEED DEFAULT TEMPLATE (All Document Heads)
-- =====================================================

INSERT INTO company_templates (template_name, description, is_active, created_by)
VALUES ('Default - All Document Heads', 'Includes all available document heads', TRUE, 1)
ON CONFLICT DO NOTHING;

-- Add all existing document heads to default template
INSERT INTO company_template_document_heads (template_id, type_id)
SELECT 
    (SELECT id FROM company_templates WHERE template_name = 'Default - All Document Heads'),
    st.id
FROM sub_type st
WHERE st.status = '1'
ON CONFLICT DO NOTHING;

-- =====================================================
-- MIGRATE EXISTING USER CSV DATA TO NEW TABLES
-- =====================================================

-- Migrate users.user_company to user_companies
INSERT INTO user_companies (user_id, company_id)
SELECT 
    u.id,
    unnest(string_to_array(u.user_company, ','))::int
FROM users u
WHERE u.user_company IS NOT NULL 
  AND u.user_company != ''
  AND u.user_company != 'NULL'
ON CONFLICT (user_id, company_id) DO NOTHING;

-- Migrate users.user_type to user_document_types
INSERT INTO user_document_types (user_id, type_id)
SELECT 
    u.id,
    unnest(string_to_array(u.user_type, ','))::int
FROM users u
WHERE u.user_type IS NOT NULL 
  AND u.user_type != ''
  AND u.user_type != 'NULL'
ON CONFLICT (user_id, type_id) DO NOTHING;

-- =====================================================
-- INITIALIZE EXISTING COMPANIES WITH ALL DOC HEADS
-- (For backward compatibility - all enabled by default)
-- =====================================================

-- Create company_authorities for existing companies (enabled for backward compat)
INSERT INTO company_authorities (company_id, authority_id, is_enabled, enabled_at)
SELECT DISTINCT
    c.id,
    a.id,
    TRUE,  -- Enabled for existing companies
    CURRENT_TIMESTAMP
FROM company c
CROSS JOIN authority a
WHERE a.status = '1'
ON CONFLICT (company_id, authority_id) DO NOTHING;

-- Create company_document_heads for existing companies (enabled for backward compat)
INSERT INTO company_document_heads (company_id, type_id, is_enabled, is_reviewed, enabled_at, custom_start_date)
SELECT 
    c.id,
    st.id,
    TRUE,  -- Enabled for existing companies
    TRUE,  -- Already reviewed
    CURRENT_TIMESTAMP,
    st.document_start_date::date
FROM company c
CROSS JOIN sub_type st
WHERE st.status = '1'
ON CONFLICT (company_id, type_id) DO NOTHING;

-- Create company_mandatory_documents for existing companies (enabled)
INSERT INTO company_mandatory_documents (company_id, type_id, document_id, is_mandatory, is_enabled, sort_order)
SELECT 
    c.id,
    md.type_id,
    md.document_id,
    CASE WHEN md.mandatory = '1' THEN TRUE ELSE FALSE END,
    TRUE,  -- Enabled for existing companies
    ROW_NUMBER() OVER (PARTITION BY c.id, md.type_id ORDER BY md.id)
FROM company c
CROSS JOIN mandatory_documents md
WHERE EXISTS (SELECT 1 FROM sub_type st WHERE st.id = md.type_id AND st.status = '1')
ON CONFLICT (company_id, type_id, document_id) DO NOTHING;

-- Create company_document_head_reminders for existing companies (enabled)
INSERT INTO company_document_head_reminders (company_id, type_id, reminder_no, days_before, 
    reminder_to_user, reminder_to_admin, reminder_to_super_admin, is_enabled)
SELECT 
    c.id,
    sr.sub_type_id,
    sr.reminder_no,
    sr.days_before,
    CASE WHEN sr.reminder_to_user = 'true' THEN TRUE ELSE FALSE END,
    CASE WHEN sr.reminder_to_admin = 'true' THEN TRUE ELSE FALSE END,
    CASE WHEN sr.reminder_to_super_admin = 'true' THEN TRUE ELSE FALSE END,
    TRUE  -- Enabled for existing companies
FROM company c
CROSS JOIN sub_type_reminders sr
WHERE EXISTS (SELECT 1 FROM sub_type st WHERE st.id = sr.sub_type_id AND st.status = '1')
ON CONFLICT (company_id, type_id, reminder_no) DO NOTHING;

-- Create company_documents for existing companies (enabled)
INSERT INTO company_documents (company_id, document_id, is_enabled, enabled_at)
SELECT DISTINCT
    c.id,
    d.id,
    TRUE,  -- Enabled for existing companies
    CURRENT_TIMESTAMP
FROM company c
CROSS JOIN documents d
WHERE d.status = '1'
ON CONFLICT (company_id, document_id) DO NOTHING;

-- Create company_document_groups for existing companies (enabled)
INSERT INTO company_document_groups (company_id, parent_document_id, child_document_ids, is_enabled)
SELECT 
    c.id,
    dg.document_id,
    dg.grouped_doc_id,
    TRUE  -- Enabled for existing companies
FROM company c
CROSS JOIN document_groups dg
ON CONFLICT (company_id, parent_document_id) DO NOTHING;

