<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Permission Model
 * 
 * Handles database queries for role-based access control.
 * 
 * @version 1.0
 * @date 2026-01-15
 */
class Permission_model extends CI_Model {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Check if a role has a specific permission
     * 
     * @param int $role_id The role ID
     * @param string $permission_key The permission key
     * @return bool True if allowed
     */
    public function has_permission($role_id, $permission_key) {
        $result = $this->db->select('is_allowed')
            ->where('role_id', $role_id)
            ->where('permission_key', $permission_key)
            ->get('role_permissions')
            ->row();
        
        return $result && $result->is_allowed;
    }

    /**
     * Get all permissions for a role
     * 
     * @param int $role_id The role ID
     * @return array Array of permission keys that are allowed
     */
    public function get_role_permissions($role_id) {
        $results = $this->db->select('permission_key')
            ->where('role_id', $role_id)
            ->where('is_allowed', true)
            ->get('role_permissions')
            ->result();
        
        return array_column($results, 'permission_key');
    }

    /**
     * Check if user has access to a company
     * 
     * @param int $user_id The user ID
     * @param int $company_id The company ID
     * @return bool True if user has access
     */
    public function user_has_company_access($user_id, $company_id) {
        $count = $this->db->where('user_id', $user_id)
            ->where('company_id', $company_id)
            ->count_all_results('user_companies');
        
        return $count > 0;
    }

    /**
     * Get all company IDs (for Super Admin)
     * 
     * @return array Array of company IDs
     */
    public function get_all_company_ids() {
        $results = $this->db->select('id')
            ->get('company')
            ->result();
        
        return array_column($results, 'id');
    }

    /**
     * Get company IDs assigned to a user
     * 
     * @param int $user_id The user ID
     * @return array Array of company IDs
     */
    public function get_user_company_ids($user_id) {
        $results = $this->db->select('company_id')
            ->where('user_id', $user_id)
            ->get('user_companies')
            ->result();
        
        return array_column($results, 'company_id');
    }

    /**
     * Get all authority IDs
     * 
     * @return array Array of authority IDs
     */
    public function get_all_authority_ids() {
        $results = $this->db->select('id')
            ->where('status', '1')
            ->get('authority')
            ->result();
        
        return array_column($results, 'id');
    }

    /**
     * Get enabled authorities for a company
     * 
     * @param int $company_id The company ID
     * @return array Array of authority IDs
     */
    public function get_enabled_authorities_for_company($company_id) {
        $results = $this->db->select('authority_id')
            ->where('company_id', $company_id)
            ->where('is_enabled', true)
            ->get('company_authorities')
            ->result();
        
        return array_column($results, 'authority_id');
    }

    /**
     * Get authority IDs assigned to a user
     * 
     * @param int $user_id The user ID
     * @return array Array of authority IDs
     */
    public function get_user_authority_ids($user_id) {
        $results = $this->db->select('authority_id')
            ->where('user_id', $user_id)
            ->get('user_authorities')
            ->result();
        
        return array_column($results, 'authority_id');
    }

    /**
     * Get all document type IDs
     * 
     * @return array Array of type IDs
     */
    public function get_all_document_type_ids() {
        $results = $this->db->select('id')
            ->where('status', '1')
            ->get('sub_type')
            ->result();
        
        return array_column($results, 'id');
    }

    /**
     * Get enabled document types for a company
     * 
     * @param int $company_id The company ID
     * @return array Array of type IDs
     */
    public function get_enabled_document_types_for_company($company_id) {
        $results = $this->db->select('type_id')
            ->where('company_id', $company_id)
            ->where('is_enabled', true)
            ->get('company_document_heads')
            ->result();
        
        return array_column($results, 'type_id');
    }

    /**
     * Get document type IDs assigned to a user
     * 
     * @param int $user_id The user ID
     * @return array Array of type IDs
     */
    public function get_user_document_type_ids($user_id) {
        $results = $this->db->select('type_id')
            ->where('user_id', $user_id)
            ->get('user_document_types')
            ->result();
        
        return array_column($results, 'type_id');
    }

    /**
     * Get companies accessible by a user with full details
     * 
     * @param int $user_id The user ID
     * @param int $role_id The user's role ID
     * @return array Array of company objects
     */
    public function get_accessible_companies($user_id, $role_id) {
        if ($role_id == '1') {
            // Super Admin gets all companies
            return $this->db->get('company')->result();
        }
        
        // Other roles get assigned companies
        return $this->db->select('c.*')
            ->from('company c')
            ->join('user_companies uc', 'c.id = uc.company_id')
            ->where('uc.user_id', $user_id)
            ->get()
            ->result();
    }

    /**
     * Get enabled document heads for a company
     * 
     * @param int $company_id The company ID
     * @param int|null $authority_id Optional authority filter
     * @return array Array of document head objects with company settings
     */
    public function get_company_document_heads($company_id, $authority_id = null) {
        $this->db->select("st.*, cdh.custom_start_date, cdh.is_enabled, cdh.is_reviewed,
            COALESCE(NULLIF(cdh.custom_start_date::text, ''), NULLIF(st.document_start_date::text, '')) as effective_start_date,
            COALESCE(NULLIF(cdh.custom_frequency_start_date::text, ''), NULLIF(st.frequency_start_date, '')) as effective_frequency_start_date,
            COALESCE(NULLIF(cdh.custom_due_in_same_next_month::text, ''), NULLIF(st.due_in_same_next_month::text, '')) as effective_due_in_same_next_month",false)
            ->from('sub_type st')
            ->join('company_document_heads cdh', 'st.id = cdh.type_id AND cdh.company_id = ' . (int)$company_id, 'inner')
            ->where('st.status', '1')
            ->where('cdh.is_enabled', true);
        
        if ($authority_id) {
            $this->db->where('st.authority_id', $authority_id);
        }
        
        return $this->db->get()->result();
    }

    /**
     * Get enabled documents for a company and document type
     * 
     * @param int $company_id The company ID
     * @param int $type_id The document type ID
     * @return array Array of document objects with company settings
     */
    public function get_company_documents($company_id, $type_id) {
        return $this->db->select('d.*, cmd.is_mandatory, cmd.sort_order')
            ->from('documents d')
            ->join('company_mandatory_documents cmd', 
                'd.id = cmd.document_id AND cmd.company_id = ' . (int)$company_id . ' AND cmd.type_id = ' . (int)$type_id, 
                'inner')
            ->where('d.status', '1')
            ->where('cmd.is_enabled', true)
            ->order_by('cmd.sort_order', 'ASC')
            ->get()
            ->result();
    }

    /**
     * Get authorities with enabled document heads for a company
     * 
     * @param int $company_id The company ID
     * @return array Array of authority objects
     */
    public function get_company_authorities($company_id) {
        return $this->db->select('a.*')
            ->from('authority a')
            ->join('company_authorities ca', 'a.id = ca.authority_id')
            ->where('ca.company_id', $company_id)
            ->where('ca.is_enabled', true)
            ->where('a.status', '1')
            ->get()
            ->result();
    }

    /**
     * Activate a document head for a company
     * Cascades to enable parent authority and child documents
     * 
     * @param int $company_id The company ID
     * @param int $type_id The document type ID
     * @param string $custom_start_date The company-specific start date
     * @param int $enabled_by User ID who enabled it
     * @return array Success/error response
     */
    public function activate_document_head($company_id, $type_id, $custom_start_date, $enabled_by) {
        // Validate start date against company start date
        $company = $this->db->get_where('company', ['id' => $company_id])->row();
        if (!$company) {
            return ['success' => false, 'message' => 'Company not found'];
        }
        
        if ($custom_start_date && strtotime($custom_start_date) < strtotime($company->co_start_date)) {
            return [
                'success' => false, 
                'message' => 'Document start date cannot be earlier than company start date (' . date('d-m-Y', strtotime($company->co_start_date)) . ')'
            ];
        }
        
        // Get document head info
        $doc_head = $this->db->get_where('sub_type', ['id' => $type_id])->row();
        if (!$doc_head) {
            return ['success' => false, 'message' => 'Document head not found'];
        }
        
        $this->db->trans_start();
        
        // 1. Enable the document head
        $this->db->where(['company_id' => $company_id, 'type_id' => $type_id])
            ->update('company_document_heads', [
                'is_enabled' => true,
                'is_reviewed' => true,
                'custom_start_date' => $custom_start_date,
                'enabled_at' => date('Y-m-d H:i:s'),
                'enabled_by' => $enabled_by,
                'updated_at' => date('Y-m-d H:i:s')
            ]);
        
        // 2. Enable parent authority
        $authority_id = $doc_head->authority_id;
        $this->db->query("
            INSERT INTO company_authorities (company_id, authority_id, is_enabled, enabled_at, enabled_by)
            VALUES (?, ?, TRUE, NOW(), ?)
            ON CONFLICT (company_id, authority_id) 
            DO UPDATE SET is_enabled = TRUE, enabled_at = NOW(), enabled_by = ?
        ", [$company_id, $authority_id, $enabled_by, $enabled_by]);
        
        // 3. Enable child mandatory documents
        $this->db->where(['company_id' => $company_id, 'type_id' => $type_id])
            ->update('company_mandatory_documents', ['is_enabled' => true]);
        
        // 4. Enable documents in company_documents
        $this->db->query("
            INSERT INTO company_documents (company_id, document_id, is_enabled, enabled_at)
            SELECT ?, document_id, TRUE, NOW()
            FROM company_mandatory_documents
            WHERE company_id = ? AND type_id = ?
            ON CONFLICT (company_id, document_id) 
            DO UPDATE SET is_enabled = TRUE, enabled_at = NOW()
        ", [$company_id, $company_id, $type_id]);
        
        // 5. Enable reminders
        $this->db->where(['company_id' => $company_id, 'type_id' => $type_id])
            ->update('company_document_head_reminders', ['is_enabled' => true]);
        
        // 6. Enable document groups
        $doc_ids = $this->db->select('document_id')
            ->where(['company_id' => $company_id, 'type_id' => $type_id])
            ->get('company_mandatory_documents')
            ->result_array();
        
        if (!empty($doc_ids)) {
            $doc_id_list = array_column($doc_ids, 'document_id');
            $this->db->where('company_id', $company_id)
                ->where_in('parent_document_id', $doc_id_list)
                ->update('company_document_groups', ['is_enabled' => true]);
        }
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            return ['success' => false, 'message' => 'Database error occurred'];
        }
        
        return ['success' => true, 'message' => 'Document head activated successfully'];
    }

    /**
     * Deactivate a document head for a company
     * 
     * @param int $company_id The company ID
     * @param int $type_id The document type ID
     * @return array Success/error response
     */
    public function deactivate_document_head($company_id, $type_id) {
        // Get document head info
        $doc_head = $this->db->get_where('sub_type', ['id' => $type_id])->row();
        if (!$doc_head) {
            return ['success' => false, 'message' => 'Document head not found'];
        }
        
        $authority_id = $doc_head->authority_id;
        
        $this->db->trans_start();
        
        // 1. Disable the document head
        $this->db->where(['company_id' => $company_id, 'type_id' => $type_id])
            ->update('company_document_heads', [
                'is_enabled' => false,
                'updated_at' => date('Y-m-d H:i:s')
            ]);
        
        // 2. Disable related mandatory documents
        $this->db->where(['company_id' => $company_id, 'type_id' => $type_id])
            ->update('company_mandatory_documents', ['is_enabled' => false]);
        
        // 3. Disable reminders
        $this->db->where(['company_id' => $company_id, 'type_id' => $type_id])
            ->update('company_document_head_reminders', ['is_enabled' => false]);
        
        // 4. Check if authority has other enabled doc heads
        $other_enabled = $this->db->where('company_id', $company_id)
            ->where('is_enabled', true)
            ->join('sub_type st', 'st.id = company_document_heads.type_id')
            ->where('st.authority_id', $authority_id)
            ->count_all_results('company_document_heads');
        
        // 5. If no other doc heads use this authority, disable it
        if ($other_enabled == 0) {
            $this->db->where(['company_id' => $company_id, 'authority_id' => $authority_id])
                ->update('company_authorities', ['is_enabled' => false]);
        }
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            return ['success' => false, 'message' => 'Database error occurred'];
        }
        
        return ['success' => true, 'message' => 'Document head deactivated successfully'];
    }

    /**
     * Get all available templates
     * 
     * @return array Array of template objects
     */
    public function get_templates() {
        return $this->db->where('is_active', true)
            ->get('company_templates')
            ->result();
    }

    /**
     * Copy template to company (all items disabled)
     * 
     * @param int $company_id The company ID
     * @param int $template_id The template ID
     * @param int $created_by User ID creating this
     * @return bool Success status
     */
    public function apply_template_to_company($company_id, $template_id, $created_by) {
        $this->db->trans_start();
        
        // 1. Copy authorities as DISABLED
        $this->db->query("
            INSERT INTO company_authorities (company_id, authority_id, is_enabled)
            SELECT DISTINCT ?, CAST(st.authority_id AS INT), FALSE
            FROM company_template_document_heads ctdh
            JOIN sub_type st ON st.id = ctdh.type_id
            WHERE ctdh.template_id = ?
            ON CONFLICT (company_id, authority_id) DO NOTHING
        ", [$company_id, $template_id]);
        
        // 2. Copy document heads as DISABLED
        $this->db->query("
            INSERT INTO company_document_heads (company_id, type_id, is_enabled, is_reviewed, created_by)
            SELECT ?, ctdh.type_id, FALSE, FALSE, ?
            FROM company_template_document_heads ctdh
            WHERE ctdh.template_id = ?
            ON CONFLICT (company_id, type_id) DO NOTHING
        ", [$company_id, $created_by, $template_id]);
        
        // 3. Copy mandatory documents as DISABLED
        $this->db->query("
            INSERT INTO company_mandatory_documents (company_id, type_id, document_id, is_mandatory, is_enabled, sort_order)
            SELECT ?, md.type_id, md.document_id, 
                CASE WHEN md.mandatory = '1' THEN TRUE ELSE FALSE END,
                FALSE,
                ROW_NUMBER() OVER (PARTITION BY md.type_id ORDER BY md.id)
            FROM mandatory_documents md
            JOIN company_template_document_heads ctdh ON md.type_id = ctdh.type_id
            WHERE ctdh.template_id = ?
            ON CONFLICT (company_id, type_id, document_id) DO NOTHING
        ", [$company_id, $template_id]);
        
        // 4. Copy reminders as DISABLED
        $this->db->query("
            INSERT INTO company_document_head_reminders (company_id, type_id, reminder_no, days_before, 
                reminder_to_user, reminder_to_admin, reminder_to_super_admin, is_enabled)
            SELECT ?, sr.sub_type_id, sr.reminder_no, sr.days_before,
                CASE WHEN sr.reminder_to_user = 'true' THEN TRUE ELSE FALSE END,
                CASE WHEN sr.reminder_to_admin = 'true' THEN TRUE ELSE FALSE END,
                CASE WHEN sr.reminder_to_super_admin = 'true' THEN TRUE ELSE FALSE END,
                FALSE
            FROM sub_type_reminders sr
            JOIN company_template_document_heads ctdh ON sr.sub_type_id = ctdh.type_id
            WHERE ctdh.template_id = ?
            ON CONFLICT (company_id, type_id, reminder_no) DO NOTHING
        ", [$company_id, $template_id]);
        
        // 5. Copy documents as DISABLED
        $this->db->query("
            INSERT INTO company_documents (company_id, document_id, is_enabled)
            SELECT DISTINCT ?, md.document_id, FALSE
            FROM mandatory_documents md
            JOIN company_template_document_heads ctdh ON md.type_id = ctdh.type_id
            WHERE ctdh.template_id = ?
            ON CONFLICT (company_id, document_id) DO NOTHING
        ", [$company_id, $template_id]);
        
        // 6. Copy document groups as DISABLED
        $this->db->query("
            INSERT INTO company_document_groups (company_id, parent_document_id, child_document_ids, is_enabled)
            SELECT ?, dg.document_id::int, dg.grouped_doc_id, FALSE
            FROM document_groups dg
            WHERE dg.document_id IS NOT NULL 
              AND dg.document_id != ''
              AND dg.document_id ~ '^\d+$'
              AND dg.document_id::int IN (
                SELECT DISTINCT md.document_id
                FROM mandatory_documents md
                JOIN company_template_document_heads ctdh ON md.type_id = ctdh.type_id
                WHERE ctdh.template_id = ?
              )
            ON CONFLICT (company_id, parent_document_id) DO NOTHING
        ", [$company_id, $template_id]);
        
        $this->db->trans_complete();
        
        return $this->db->trans_status();
    }
    
    /**
     * Count pending (disabled) document heads for a company
     * 
     * @param int $company_id The company ID
     * @return int Count of pending document heads
     */
    public function count_pending_doc_heads($company_id) {
        // Count disabled document heads for this company
        $count = $this->db->where('company_id', $company_id)
            ->where('is_enabled', false)
            ->count_all_results('company_document_heads');
        
        return $count;
    }
    
    /**
     * Count new document heads in master that are not yet added to company
     * 
     * @param int $company_id The company ID
     * @return int Count of new document heads not in company
     */
    public function count_new_master_doc_heads($company_id) {
        // Count document heads in master (sub_type) that are not in company_document_heads
        $sql = "SELECT COUNT(*) as cnt 
                FROM sub_type st 
                WHERE st.status = '1' 
                  AND st.id NOT IN (
                      SELECT type_id FROM company_document_heads WHERE company_id = ?
                  )";
        
        $result = $this->db->query($sql, [$company_id])->row();
        
        return $result ? (int)$result->cnt : 0;
    }
    
    /**
     * Get total pending items count for company doc heads
     * (disabled items + new master items not added)
     * 
     * @param int $company_id The company ID
     * @return int Total pending count
     */
    public function get_pending_doc_heads_count($company_id) {
        $disabled_count = $this->count_pending_doc_heads($company_id);
        $new_count = $this->count_new_master_doc_heads($company_id);
        
        return $disabled_count + $new_count;
    }
    
    /**
     * Get pending doc heads count for all companies user has access to
     * 
     * @param array $company_ids Array of company IDs user can access
     * @return int Total pending count across all accessible companies
     */
    public function get_total_pending_doc_heads($company_ids) {
        if (empty($company_ids)) {
            return 0;
        }
        
        $total = 0;
        foreach ($company_ids as $company_id) {
            $total += $this->get_pending_doc_heads_count($company_id);
        }
        
        return $total;
    }
}

