<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Security Helper for URL encryption/decryption
 * Used to secure sensitive parameters in URLs
 */

/**
 * Encrypt a value for use in URLs
 * @param mixed $value The value to encrypt
 * @return string URL-safe encrypted string
 */
if (!function_exists('encrypt_url')) {
    function encrypt_url($value) {
        $CI =& get_instance();
        $CI->load->library('encryption');
        
        $encrypted = $CI->encryption->encrypt($value);
        // Make it URL safe
        return rtrim(strtr(base64_encode($encrypted), '+/', '-_'), '=');
    }
}

/**
 * Decrypt a URL-encrypted value
 * @param string $encrypted The encrypted string from URL
 * @return mixed The decrypted value
 */
if (!function_exists('decrypt_url')) {
    function decrypt_url($encrypted) {
        $CI =& get_instance();
        $CI->load->library('encryption');
        
        // Reverse URL-safe encoding
        $encrypted = base64_decode(strtr($encrypted, '-_', '+/'));
        return $CI->encryption->decrypt($encrypted);
    }
}

/**
 * Generate a secure token for one-time use links
 * @param int $length Length of the token
 * @return string Random token
 */
if (!function_exists('generate_secure_token')) {
    function generate_secure_token($length = 32) {
        return bin2hex(random_bytes($length / 2));
    }
}

/**
 * Validate that a user has access to a specific resource
 * @param int $resource_id The resource ID to check
 * @param string $resource_type Type of resource (company, document, etc.)
 * @return bool True if access is allowed
 */
if (!function_exists('validate_access')) {
    function validate_access($resource_id, $resource_type = 'company') {
        $CI =& get_instance();
        $user_id = $CI->session->userdata('id');
        
        if (!$user_id) {
            return false;
        }
        
        $CI->load->model('Dashboard_model');
        $role_id = $CI->Dashboard_model->getUserRole($user_id);
        
        // Super admins and admins have access to everything
        if ($CI->Dashboard_model->isAdmin($role_id)) {
            return true;
        }
        
        // For regular users, check if they have access to this resource
        if ($resource_type === 'company') {
            $allowed = $CI->Dashboard_model->getUserAllowedCompanyIds($user_id, $role_id);
            return in_array($resource_id, $allowed);
        }
        
        return false;
    }
}

/**
 * Sanitize input parameter
 * @param mixed $input The input to sanitize
 * @param string $type Type of sanitization (int, string, email)
 * @return mixed Sanitized value
 */
if (!function_exists('sanitize_input')) {
    function sanitize_input($input, $type = 'string') {
        if ($input === null || $input === '') {
            return null;
        }
        
        switch ($type) {
            case 'int':
                return filter_var($input, FILTER_VALIDATE_INT) !== false 
                    ? (int) $input 
                    : null;
            
            case 'email':
                return filter_var($input, FILTER_VALIDATE_EMAIL) 
                    ? $input 
                    : null;
            
            case 'url':
                return filter_var($input, FILTER_VALIDATE_URL) 
                    ? $input 
                    : null;
            
            case 'string':
            default:
                return htmlspecialchars(strip_tags(trim($input)), ENT_QUOTES, 'UTF-8');
        }
    }
}

