# Dashboard Frequency-Based Document Management

## Overview

The Dashboard module manages document tracking across different submission frequencies (Monthly, Quarterly, Half-Yearly, Yearly) with support for company-specific fiscal years.

---

## Key Database Columns

### `sub_type` Table
| Column | Description | Example |
|--------|-------------|---------|
| `frequency` | Submission frequency | `Monthly`, `Quarterly`, `Half Yearly`, `Yearly` |
| `frequency_start_date` | Due day of month (1-31) | `7` (due on 7th) |
| `document_start_date` | Date from which document tracking begins | `2025-04-01` |
| `due_in_same_next_month` | When document is due: `0`=same month, `1`=next month | `0` or `1` |

### `company` Table
| Column | Description | Example |
|--------|-------------|---------|
| `fiscal_year` | Company's fiscal year cycle | `apr-mar`, `oct-sep`, `jul-jun`, `jan-dec` |

---

## Frequency Types

### 1. Monthly
- **Period:** Every month
- **Due Date:** `frequency_start_date` of each month
- **Example:** If `frequency_start_date = 7` and `document_start_date = 2025-04-01`
  - First due: April 7, 2025
  - Next dues: May 7, June 7, July 7, etc.

### 2. Quarterly (4 times per fiscal year)
- **Period:** Every 3 months based on fiscal year
- **Example for Apr-Mar fiscal year:**
  - Q1: Apr-Jun (due in Jun or Jul based on `due_in_same_next_month`)
  - Q2: Jul-Sep (due in Sep or Oct)
  - Q3: Oct-Dec (due in Dec or Jan)
  - Q4: Jan-Mar (due in Mar or Apr)

### 3. Half-Yearly (2 times per fiscal year)
- **Period:** Every 6 months based on fiscal year
- **Example for Apr-Mar fiscal year:**
  - H1: Apr-Sep (due in Sep or Oct)
  - H2: Oct-Mar (due in Mar or Apr)

### 4. Yearly (1 time per fiscal year)
- **Period:** Once per fiscal year
- **Example for Apr-Mar fiscal year:**
  - Due at end of fiscal year: March (or April if `due_in_same_next_month = 1`)

---

## Fiscal Year Support

The system supports any 12-month fiscal year cycle. The fiscal year is defined in the `company` table's `fiscal_year` column.

### Supported Patterns

| Fiscal Year Pattern | Start Month | Quarter End Months |
|---------------------|-------------|-------------------|
| `jan-dec` | January (1) | Mar, Jun, Sep, Dec |
| `feb-jan` | February (2) | Apr, Jul, Oct, Jan |
| `mar-feb` | March (3) | May, Aug, Nov, Feb |
| `apr-mar` | April (4) | Jun, Sep, Dec, Mar |
| `may-apr` | May (5) | Jul, Oct, Jan, Apr |
| `jun-may` | June (6) | Aug, Nov, Feb, May |
| `jul-jun` | July (7) | Sep, Dec, Mar, Jun |
| `aug-jul` | August (8) | Oct, Jan, Apr, Jul |
| `sep-aug` | September (9) | Nov, Feb, May, Aug |
| `oct-sep` | October (10) | Dec, Mar, Jun, Sep |
| `nov-oct` | November (11) | Jan, Apr, Jul, Oct |
| `dec-nov` | December (12) | Feb, May, Aug, Nov |

### Parsing Logic
The system parses fiscal year strings like:
- Full names: `january-december`, `april-march`
- Abbreviations: `jan-dec`, `apr-mar`, `oct-sep`
- Numeric: `4-3`, `10-9`, `04-03`

---

## Dashboard Tabs Logic

### Pending Tab
Documents where:
- Due date is in current month AND due day >= today
- Document period is on or after `document_start_date`
- Not yet uploaded

### Overdue Tab
Documents where:
- Due date has passed (due day < today OR previous months)
- Document period is on or after `document_start_date`
- Not yet uploaded

### Upcoming Tab
Documents where:
- Due date is in next month
- Document period is on or after `document_start_date`
- Not yet uploaded

### Uploaded Tab
All uploaded documents grouped by document head and period.

---

## Code Structure

### Model Functions (`Dashboard_model.php`)

```
getPendingDocuments()
├── Monthly frequency: Direct SQL query
└── _getNonMonthlyPending(): Quarterly, Half-Yearly, Yearly

getOverdueDocuments()
├── Monthly (current month): Direct SQL query
├── Monthly (previous months): _getPreviousMonthsOverdue()
└── _getNonMonthlyOverdue(): Quarterly, Half-Yearly, Yearly

getUpcomingDocuments()
├── Monthly frequency: Direct SQL query
└── _getNonMonthlyUpcoming(): Quarterly, Half-Yearly, Yearly
```

### Helper Functions

| Function | Purpose |
|----------|---------|
| `_getPeriodsForFrequency()` | Calculate period end months for non-monthly frequencies based on fiscal year |
| `_getFiscalStartMonth()` | Parse fiscal year string to start month number |
| `_isDocumentUploaded()` | Check if document is uploaded for specific period |
| `_groupDocumentsWithStatus()` | Group documents with upload status |
| `_groupDocumentsWithStatusAndPeriod()` | Group documents with status and period info |

---

## Document Start Date Logic

Documents only appear in dashboard if their period is **on or after** `document_start_date`:

```
Period Month/Year >= Document Start Date Month/Year
```

**Example:**
- `document_start_date = 2025-04-01`
- January 2025 period: ❌ NOT shown (before start date)
- February 2025 period: ❌ NOT shown (before start date)
- March 2025 period: ❌ NOT shown (before start date)
- April 2025 period: ✅ Shown (on or after start date)
- May 2025 period: ✅ Shown

### SQL Implementation
```sql
AND (
    st.document_start_date IS NULL 
    OR TRIM(st.document_start_date::text) = ''
    OR (
        MAKE_DATE(doc_period_year, doc_period_month, 1) 
        >= DATE_TRUNC('month', st.document_start_date::date)
    )
)
```

---

## Due In Same/Next Month Logic

| Value | Meaning | Example |
|-------|---------|---------|
| `0` | Due in same month as period | March period → Due in March |
| `1` | Due in next month after period | March period → Due in April |

**Example with Monthly Frequency:**
- Period: December 2025
- `frequency_start_date = 7` (due on 7th)
- `due_in_same_next_month = 0`: Due on December 7, 2025
- `due_in_same_next_month = 1`: Due on January 7, 2026

---

## Period Calculation for Non-Monthly Frequencies

### Quarterly (Fiscal Year: Apr-Mar)
```
Fiscal Start = 4 (April)

Q1: End Month = (4-1 + 3) % 12 = 6 (June)
Q2: End Month = (4-1 + 6) % 12 = 9 (September)
Q3: End Month = (4-1 + 9) % 12 = 0 → 12 (December)
Q4: End Month = (4-1 + 12) % 12 = 3 (March)
```

### Quarterly (Fiscal Year: Oct-Sep)
```
Fiscal Start = 10 (October)

Q1: End Month = (10-1 + 3) % 12 = 0 → 12 (December)
Q2: End Month = (10-1 + 6) % 12 = 3 (March)
Q3: End Month = (10-1 + 9) % 12 = 6 (June)
Q4: End Month = (10-1 + 12) % 12 = 9 (September)
```

### Half-Yearly (Fiscal Year: Jul-Jun)
```
Fiscal Start = 7 (July)

H1: End Month = (7-1 + 6) % 12 = 0 → 12 (December)
H2: End Month = (7-1 + 12) % 12 = 6 (June)
```

### Yearly (Fiscal Year: Apr-Mar)
```
Fiscal Start = 4 (April)
End Month = 4 - 1 = 3 (March)
```

---

## Architecture Diagram

```
┌─────────────────────────────────────────────────────────────┐
│                      Dashboard View                          │
├──────────┬──────────┬──────────┬──────────┬────────────────┤
│ Favourite│ Pending  │ Overdue  │ Upcoming │ Uploaded       │
└────┬─────┴────┬─────┴────┬─────┴────┬─────┴───────┬────────┘
     │          │          │          │             │
     ▼          ▼          ▼          ▼             ▼
┌─────────────────────────────────────────────────────────────┐
│                   Dashboard Controller                       │
│                   (dashboard.php)                            │
└─────────────────────────┬───────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────────┐
│                    Dashboard_model.php                       │
├─────────────────────────────────────────────────────────────┤
│  Public Methods:                                             │
│  • getPendingDocuments()    • getOverdueDocuments()         │
│  • getUpcomingDocuments()   • getUploadedDocumentsGrouped() │
│  • getFavourites()          • getDashboardCounts()          │
├─────────────────────────────────────────────────────────────┤
│  Private Helper Methods:                                     │
│  • _getNonMonthlyPending()  • _getNonMonthlyOverdue()       │
│  • _getNonMonthlyUpcoming() • _getPreviousMonthsOverdue()   │
│  • _getPeriodsForFrequency()• _getFiscalStartMonth()        │
│  • _isDocumentUploaded()    • _groupDocumentsWithStatus()   │
└─────────────────────────────────────────────────────────────┘
                          │
                          ▼
┌─────────────────────────────────────────────────────────────┐
│                     PostgreSQL Database                      │
├──────────┬──────────┬──────────┬──────────┬────────────────┤
│ sub_type │ company  │ documents│ uploaded │ favourites     │
│          │          │          │_documents│                │
├──────────┼──────────┼──────────┼──────────┼────────────────┤
│frequency │fiscal_   │document_ │type_id   │upload_id       │
│frequency_│year      │name      │document_ │user_id         │
│start_date│          │          │id        │created_at      │
│document_ │          │          │company_id│                │
│start_date│          │          │document_ │                │
│due_in_   │          │          │month     │                │
│same_next_│          │          │document_ │                │
│month     │          │          │year      │                │
└──────────┴──────────┴──────────┴──────────┴────────────────┘
```

---

## Example Scenarios

### Scenario 1: Monthly Document
**Configuration:**
- `frequency = 'Monthly'`
- `frequency_start_date = 7`
- `document_start_date = 2025-04-01`
- `due_in_same_next_month = 0`

**Result:**
- First due date: April 7, 2025
- Subsequent: May 7, June 7, July 7, etc.
- March 2025 and earlier: NOT shown

### Scenario 2: Quarterly Document (Apr-Mar FY)
**Configuration:**
- `frequency = 'Quarterly'`
- `frequency_start_date = 15`
- `document_start_date = 2025-04-01`
- `due_in_same_next_month = 1`
- `fiscal_year = 'apr-mar'`

**Result:**
- Q1 (Apr-Jun): Due July 15, 2025
- Q2 (Jul-Sep): Due October 15, 2025
- Q3 (Oct-Dec): Due January 15, 2026
- Q4 (Jan-Mar): Due April 15, 2026

### Scenario 3: Half-Yearly Document (Oct-Sep FY)
**Configuration:**
- `frequency = 'Half Yearly'`
- `frequency_start_date = 10`
- `document_start_date = 2024-10-01`
- `due_in_same_next_month = 0`
- `fiscal_year = 'oct-sep'`

**Result:**
- H1 (Oct-Mar): Due March 10
- H2 (Apr-Sep): Due September 10

### Scenario 4: Yearly Document (Jul-Jun FY)
**Configuration:**
- `frequency = 'Yearly'`
- `frequency_start_date = 30`
- `document_start_date = 2024-07-01`
- `due_in_same_next_month = 1`
- `fiscal_year = 'jul-jun'`

**Result:**
- FY 2024-25: Due July 30, 2025 (next month after June)

---

## Troubleshooting

### Issue: Documents not appearing in dashboard
**Check:**
1. Is `document_start_date` in the future?
2. Is the `frequency` value correct (case-insensitive)?
3. Is `frequency_start_date` a valid number (1-31)?
4. Is the company's `fiscal_year` correctly set?

### Issue: Wrong quarter/period calculation
**Check:**
1. Verify `fiscal_year` format in company table
2. Check if the first month abbreviation is correct
3. Supported formats: `apr-mar`, `oct-sep`, `jul-jun`, etc.

### Issue: Document showing in wrong tab
**Check:**
1. Verify current date vs due date
2. Check `due_in_same_next_month` value (0 or 1)
3. Verify document is not already uploaded

---

## Version History

| Date | Version | Changes |
|------|---------|---------|
| 2025-12-25 | 1.0 | Initial frequency-based implementation |
| 2025-12-25 | 1.1 | Added support for all fiscal year patterns (jan-dec through dec-nov) |
| 2025-12-25 | 1.2 | Added _getNonMonthlyOverdue and _getNonMonthlyUpcoming functions |
| 2025-12-25 | 1.3 | Enhanced document_start_date validation for all frequencies |

---

## Related Files

| File | Purpose |
|------|---------|
| `application/models/Dashboard_model.php` | Main model with all frequency logic |
| `application/controllers/dashboard.php` | Controller handling dashboard requests |
| `application/views/dashboard.php` | Dashboard view with tabs |
| `docs/DASHBOARD_FEATURE_DOCUMENT.md` | General dashboard feature documentation |

---

## Author
Generated for DMS Project - Document Management System

